/*
    Electrical Networks - Electrical network calculations
    Copyright © 2013 Harry Whitfield

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.    See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA

    Electrical Networks - version 2.1
    27 February, 2013
    Copyright 2013 Harry Whitfield
    mailto:g6auc@arrl.net
*/

/*global NETWORK, buildVitality, gErrMess:true, formPwr, network, network_output,
    gain, debug, MenuArray, unit
*/

/*properties
    LN10, LOG10E, abs, attenB, attenBT, attenL, attenLL, attenLS, attenPI,
    attenT, exp, fixed, floor, gain, length, log, p10, p12, p20, p23, p30, p31,
    p34, p41, pOut, prototype, r10, r12, r20, r23, r30, r31, r34, r41, src,
    toFixed, value, zIn, zOut
*/

//include("network.js");

var rb = 5; // digits before the decimal point - resistance
var ra = 2; // digits after  the decimal point - resistance

var pb = 5; // digits before the decimal point - power in resistor
var pa = 2; // digits after  the decimal point - power in resistor

var qb = 5; // digits before the decimal point - power out
var qa = 2; // digits after  the decimal point - power out

var gb = 5; // digits before the decimal point - gain
var ga = 5; // digits after  the decimal point - gain

var eol = "\n";
//if (system.platform === "windows") { eol = "\r\n"; }

var gNetwork_output = "";

Number.prototype.fixed = function (b, a) {
    var n, pad;

    if (isNaN(this)) {
        n = "NaN";
        while (n.length < b + a + 1) { n = " " + n; }
        return n;
    }

    n = String(Math.floor(Math.abs(Number(this.toFixed(a))))).length;
    if (this < 0) { n += 1; }
    pad = "";
    while (n < b) { pad += " "; n += 1; }
    return pad + this.toFixed(a);
};

function log10(x) {
    return Math.LOG10E * Math.log(x);
}

function exp10(x) {
    return Math.exp(Math.LN10 * x);
}

function check(mode, rIn, rOut, power, gain) {
    if (typeof mode !== "number") { return false; }

    if (typeof rIn !== "number") { return false; }
    if (isNaN(rIn)) { return false; }
    if (rIn <= 0) { return false; }

    if (typeof rOut !== "number") { return false; }
    if (isNaN(rOut)) { return false; }
    if (rOut <= 0) { return false; }

    if (typeof power !== "number") { return false; }
    if (power < 0) { return false; }
    if (isNaN(power)) { return false; }

    if (typeof gain !== "number") { return false; }
    if (isNaN(gain)) { return false; }
    if (gain > 1) { return false; }
    return true;
}

function theNetwork(mode, rIn, rOut, power, theGain) {
    var eol = "\n", s = "", t, u, oPower, Rin, Rout, R3, R4,
        attenPI, attenT, attenBT, attenB, 
//        pAttenPI, pAttenT, pAttenBT, pAttenB,
        attenL, attenLS, attenLL,
//        pAttenL, pAttenLS, pAttenLL,
        one, two, three, four, zIn, zOut, pOne, pTwo, pThree, pFour, pOut, maxGain,
        scale = MenuArray[1][unit];

    gErrMess = "Invalid data";

    if (!check(mode, rIn, rOut, power, theGain)) { return NaN; }

    switch (mode) {
    case 0:                                                         // PI-network
        attenPI = NETWORK.attenPI(rIn, rOut, theGain, power);       // (r12, r23, r31)
        one = attenPI.r31;
        two = attenPI.r23;
        three = attenPI.r12;
//      pAttenPI = NETWORK.pAttenPI(rIn, rOut, theGain, power);     // (pIn, p12, p23, p31, pOut)
		zIn = attenPI.zIn;
		zOut = attenPI.zOut;
        pOne = attenPI.p31;
        pTwo = attenPI.p23;
        pThree = attenPI.p12;
        pOut = attenPI.pOut;
        if (isNaN(three)) {	// r12, r23, r31
            maxGain = attenPI.r23;                                 // in dB
            gErrMess = "Error: Gain must be less than " + exp10(maxGain / 10).toFixed(7) + " = " + maxGain.toFixed(5) + "dB";
        }
        s += "PI-Network" + eol;                                    // (r12, r23, r31)
        Rin = "R13: ";
        Rout = "R23: ";
        R3 = "R12: ";
        break;
    case 2:                                                         // T-NETWORK
        attenT = NETWORK.attenT(rIn, rOut, theGain, power);         // (r10, r20, r30);
        one = attenT.r10;
        two = attenT.r20;
        three = attenT.r30;
//      pAttenT = NETWORK.pAttenT(rIn, rOut, theGain, power);       // (pIn, p10, p20, p30, pOut);
		zIn = attenT.zIn;
		zOut = attenT.zOut;
        pOne = attenT.p10;
        pTwo = attenT.p20;
        pThree = attenT.p30;
        pOut = attenT.pOut;
        if (isNaN(one)) {	// r10, r20, r30
            maxGain = attenT.r20;                                  // in dB
            gErrMess = "Error: Gain must be less than " + exp10(maxGain / 10).toFixed(7) + " = " + maxGain.toFixed(5) + "dB";
        }
        s += "T-Network" + eol;
        Rin = "R10: ";
        Rout = "R20: ";
        R3 = "R30: ";
        break;
    case 4:                                                         // BT-NETWORK
        attenBT = NETWORK.attenBT(rIn, theGain, power);             // (r10, r20, r30, r12)
        one = attenBT.r10;
        two = attenBT.r20;
        three = attenBT.r30;
        four = attenBT.r12;
//		pAttenBT = NETWORK.pAttenBT(theGain, power);                // (pIn, p12, p10, p20, p30, pOut)
        pOne = attenBT.p10;
        pTwo = attenBT.p20;
        pThree = attenBT.p30;
        pFour = attenBT.p12;
        pOut = attenBT.pOut;
        s += "BT-Network" + eol;
        Rin = "R10: ";
        Rout = "R20: ";
        R3 = "R30: ";
        R4 = "R12: ";
        break;
    case 5:                                                         // B-NETWORK
        attenB = NETWORK.attenB(rIn, theGain, power);               // (r12, r23, r34, r41)
        one = attenB.r12;
        two = attenB.r23;
        three = attenB.r34;
        four = attenB.r41;
//		pAttenB = NETWORK.pAttenB(theGain, power);                  // (pIn, p12, p23, p34, p41, pOut)
        pOne = attenB.p12;
        pTwo = attenB.p23;
        pThree = attenB.p34;
        pFour = attenB.p41;
        pOut = attenB.pOut;
        s += "B-Network" + eol;
        Rin = "R12: ";
        Rout = "R23: ";
        R3 = "R34: ";
        R4 = "R41: ";
        break;
    case 6:                                                         // L-NETWORK
        attenL = NETWORK.attenL(rIn, rOut, power);                   // (r10, r20, r30, gain)
        one = attenL.r10;
        two = attenL.r20;
        three = attenL.r30;
        theGain = attenL.gain;
//      pAttenL = NETWORK.pAttenL(rIn, rOut, power);                // (pIn, p10, p20, p30, pOut)
		zIn = attenL.zIn;
		zOut = attenL.zOut;
        pOne = attenL.p10;
        pTwo = attenL.p20;
        pThree = attenL.p30;
        pOut = attenL.pOut;
        if (rIn > rOut) {
            s += "L1-Network" + eol;
            Rin = "R12: ";
            Rout = "";
            R3 = "R23: ";
        } else if (rOut > rIn) {
            s += "L2-Network" + eol;
            Rin = "";
            Rout = "R12: ";
            R3 = "R13: ";
        } else {
            s += "L3-Network" + eol;
            Rin = "";
            Rout = "";
            R3 = "";
        }
        break;
    case 7:                                                         // LS-NETWORK
        attenLS = NETWORK.attenLS(rIn, rOut, theGain, power);       // (r10, r20, r30, gain)
        one = attenLS.r10;
        two = attenLS.r20;
        three = attenLS.r30;
//      pAttenLS = NETWORK.pAttenLS(rIn, rOut, theGain, power);     // (pIn, p10, p20, p30, pOut)
		zIn = attenLS.zIn;
		zOut = attenLS.zOut;
        if (rIn >= rOut) {
            pOne = attenLS.p10;
            pTwo = attenLS.p20;
            pThree = attenLS.p30;
            pOut = attenLS.pOut;
            theGain = pOut / power; // the actual gain
        } else {
            pOne = attenLS.p10;
            pTwo = attenLS.p20;
            pThree = attenLS.p30;
            pOut = attenLS.pOut;
            theGain = pOut / power; // the actual gain
        }
        if (isNaN(one)) {	// r10, r20, r30
            maxGain = attenLS.r20;                                 // in dB
            gErrMess = "Error: Gain must be less than " + exp10(maxGain / 10).toFixed(7) + " = " + maxGain.toFixed(5) + "dB";
        }
        if (rIn >= rOut) {
            s += "LS1-Network" + eol;
            Rin = "R12: ";
            Rout = "";
            R3 = "R23: ";
        } else {
            s += "LS2-Network" + eol;
            Rin = "";
            Rout = "R12: ";
            R3 = "R13: ";
        }
        break;
    case 8:                                                         // LL-NETWORK
        attenLL = NETWORK.attenLL(rIn, rOut, theGain, power);              // (r10, r20, r30, gain)
        one = attenLL.r10;
        two = attenLL.r20;
        three = attenLL.r30;
//      pAttenLL = NETWORK.pAttenLL(rIn, rOut, theGain, power);     // (pIn, p10, p20, p30, pOut)
		zIn = attenLL.zIn;
		zOut = attenLL.zOut;
        if (rIn >= rOut) {
            pOne = attenLL.p10;
            pTwo = attenLL.p20;
            pThree = attenLL.p30;
            pOut = attenLL.pOut;
            theGain = pOut / power; // the actual gain
        } else {
            pOne = attenLL.p10;
            pTwo = attenLL.p20;
            pThree = attenLL.p30;
            pOut = attenLL.pOut;
            theGain = pOut / power; // the actual gain
        }
        if (isNaN(one)) {	// 	// r10, r20, r30
            maxGain = attenLL.r20;                                 // in dB
            gErrMess = "Error: Gain must be less than " + exp10(maxGain / 10).toFixed(7) + " = " + maxGain.toFixed(5) + "dB";
        }
        if (rIn >= rOut) {
            s += "LL1-Network" + eol;
            Rin = "R12: ";
            Rout = "";
            R3 = "R23: ";
        } else {
            s += "LL2-Network" + eol;
            Rin = "";
            Rout = "R12: ";
            R3 = "R13: ";
        }
        break;
    default:
        return NaN;
    }

    t = s;
    t += "Source impedance   " + rIn.fixed(rb, ra)     + " ohms" + eol;
    t += "Load impedance     " + rOut.fixed(rb, ra)    + " ohms" + eol;
    t += "Power input        " + formPwr(power, scale) + " " + scale;
    if (scale !== 'watts') { t += "   == " + power.fixed(4, 6) + 'W'; }
    t += eol;

    u = "";

    if ((mode < 6) || (Rin  !== "")) { u += Rin  + one.fixed(rb, ra) + " ohms " + pOne.fixed(pb, pa)   + " watts" + eol; }
    if ((mode < 6) || (Rout !== "")) { u += Rout + two.fixed(rb, ra) + " ohms " + pTwo.fixed(pb, pa)   + " watts" + eol; }
    if ((mode !== 6) || (three !== Infinity)) { u += R3 + three.fixed(rb, ra) + " ohms " + pThree.fixed(pb, pa) + " watts" + eol; }

    if ((mode === 4) || (mode === 5)) {
        u += R4 + four.fixed(rb, ra) + " ohms " + pFour.fixed(pb, pa) + " watts" + eol;
    }

    s += u;
    t += u;

    oPower = pOut;

    s += "Output             " + oPower.fixed(pb, pa) + " " + "watts" + eol;
    t += "Power output       " + formPwr(oPower, scale) + " " + scale;
    if (scale !== 'watts') { t += "   == " + oPower.fixed(4, 6) + 'W'; }
    t += eol;

    u = "Gain ";
    if (theGain < 0) {
        u += exp10(theGain / 10).fixed(gb, 7) + " " + theGain.fixed(gb, ga) + " dB" + eol;
    } else {
        u += theGain.fixed(gb, 7)  + " " + (10 * log10(theGain)).fixed(gb, ga) + " dB" + eol;
    }
    t += u;

    if (debug && (mode !== 4) && (mode !== 5)) {
        u += "zIn  " + zIn.fixed(rb, ra) + " ohms " + eol;
        u += "zOut " + zOut.fixed(rb, ra) + " ohms " + eol;
    }
    s += u;

	network_output.value = s;
    gNetwork_output = t;

    if (mode === 6) {
        if (theGain < 0) {
            gain.value = exp10(theGain / 10).toFixed(7) + " = " + theGain.toFixed(5) + " dB" + eol;
        } else {
            gain.value = theGain.toFixed(7)  + " = " + (10 * log10(theGain)).toFixed(5) + " dB" + eol;
        }
    }

    if ((mode === 7) || (mode === 8)) {
        if (rIn > rOut) {
			network.src = "Resources/Lpad1.png";
        } else if (rOut > rIn) {
			network.src = "Resources/Lpad2.png";
        } else {
            if ((one === 0) && (three === Infinity)) {
				network.src = "Resources/Lpad3.png";
            } else {
				network.src = "Resources/Lpad1.png";
            }
        }
    }

    return oPower;
}

function theTable(mode, rIn, rOut, power) {
    var gain, beta, oPower, s, i, Rin, Rout, R3, Pin, Pout, P3,
    	scale = MenuArray[1][unit];

    function one(mode, Z, beta) {
        switch (mode) {
        case 1:
            return (Z * (1 + beta) / (1 - beta)).fixed(rb, ra);
        case 3:
            return (Z * (1 - beta) / (1 + beta)).fixed(rb, ra);
        case 9:
            return Z.fixed(rb, ra);
        case 10:
            return (Z * (1 - beta) / (1 + beta)).fixed(rb, ra);
        }
    }

    function two(mode, Z, beta) {
        switch (mode) {
        case 1:
            return (Z * (1 + beta) / (1 - beta)).fixed(rb, ra);
        case 3:
            return (Z * (1 - beta) / (1 + beta)).fixed(rb, ra);
        case 9:
            return (Z * beta / (1 - beta)).fixed(rb, ra);
        case 10:
            return (Z * (1 + beta) / (1 - beta)).fixed(rb, ra);
        }
    }

    function three(mode,  Z, beta) {
        switch (mode) {
        case 1:
            return (Z * (1 - beta * beta) / (2 * beta)).fixed(rb, ra);
        case 3:
            return (Z * (2 * beta) / (1 - beta * beta)).fixed(rb, ra);
        case 9:
            return (Z * (1 - beta) / beta).fixed(rb, ra);
        case 10:
            return "        ";
        }
    }

    function pOne(mode, power, beta) {
        switch (mode) {
        case 1:
        case 3:
            return (power * (1 - beta) / (1 + beta)).fixed(pb, pa);
        case 9:
            return (power * (1 - beta) * (1 - beta)).fixed(pb, pa);
        case 10:
            return (power * (1 - beta * beta) / 4).fixed(pb, pa);
        }
    }

    function pTwo(mode, power, beta) {
        switch (mode) {
        case 1:
        case 3:
            return (power * beta * beta * (1 - beta) / (1 + beta)).fixed(pb, pa);
        case 9:
            return (power * beta * (1 - beta)).fixed(pb, pa);
        case 10:
            return (power * (1 - beta * beta) / 4).fixed(pb, pa);
        }
    }

    function pThree(mode, power, beta) {
        switch (mode) {
        case 1:
        case 3:
            return (power * 2 * beta * (1 - beta) / (1 + beta)).fixed(pb, pa);
        case 9:
            return (power * beta * (1 - beta)).fixed(pb, pa);
        case 10:
            return "        ";
        }
    }

    function pOut(mode, power, beta) {
//    	alert(mode);
        return power * beta * beta;
    }

    gErrMess = "Invalid data";

    if (!check(mode, rIn, rIn, power, 0)) { return ""; }
    s = "";

    switch (mode) {
    case 1:
        s += "PI-Network" + eol;
        Rin  = "  R13  ";
        Rout = "  R23  ";
        R3   = "  R12  ";
        Pin  = "  P13  ";
        Pout = "  P23  ";
        P3   = "  P12  ";
        break;
    case 3:
        s += "T-Network" + eol;
        Rin  = "  R10  ";
        Rout = "  R20  ";
        R3   = "  R30  ";
        Pin  = "  P10  ";
        Pout = "  P20  ";
        P3   = "  P30  ";
        break;
    case 9:
        s += "BT-Network" + eol;
        Rin  = "R10&R20";
        Rout = "  R30  ";
        R3   = "  R12  ";
        Pin  = "  P10  ";
        Pout = "  P30  ";
        P3   = "  P12  ";
        break;
    case 10:
        s += "B-Network" + eol;
        Rin  = "R12&R34";
        Rout = "R23&R41";
        R3   = "  XXX  ";
        Pin  = "P12&P34";
        Pout = "P23&P41";
        P3   = "  XXX  ";
        break;
    default:
        return "";
    }

    s += eol;
    s += "Source impedance " + rIn.fixed(rb, ra)     + " ohms" + eol;
    s += "Load impedance   " + rOut.fixed(rb, ra)    + " ohms" + eol;
    s += "Power input      " + formPwr(power, scale) + " " + scale;
    if (scale !== 'watts') { s += "   == " + formPwr(power, 'watts') + ' watts'; }
    s += eol;
    if (mode === 9) {
    	s += "Power in R20     " + formPwr(0, 'watts') + " " + 'watts';
    }
    s += eol;
    s += eol;
    s += "dB   " + Rin + "  " + Pin + "  " + Rout + "  " + Pout + "  ";
	if (mode !== 10) {
    	s += R3 + "  " + P3 + "  ";
    }
    s += "PowerOut";
    if (scale !== 'watts') { s += "    " + scale; }
    s += eol;
    s += eol;
    
    for (i = 1; i < 20; i += 1) {   	// 1 to 19 dB attenuation
        gain = -1 * i;                  // in steps of 1
        beta = Math.exp(Math.log(10) * gain / 20);
        oPower = pOut(mode, power, beta);

        s += (-gain).fixed(2, 0) + " ";
        s += one(mode, rIn, beta) + " " + pOne(mode, power, beta) + " ";
        s += two(mode, rIn, beta) + " " + pTwo(mode, power, beta) + " ";
        if (mode !== 10) {
        	s += three(mode, rIn, beta) + " " + pThree(mode, power, beta) + " ";
        }
        s += formPwr(oPower, 'watts');
        if (scale !== 'watts') { s += "  " + formPwr(oPower, scale); }
        s += eol;
    }
    for (i = 4; i < 11; i += 1) {                     // 4*5=20 to 10*5=50 dB attenuation
        gain = -5 * i;                               // in steps of 5
        beta = Math.exp(Math.log(10) * gain / 20);
        oPower = pOut(mode, power, beta);

        s += (-gain).fixed(2, 0) + " ";
        s += one(mode, rIn, beta) + " " + pOne(mode, power, beta) + " ";
        s += two(mode, rIn, beta) + " " + pTwo(mode, power, beta) + " ";
        if (mode !== 10) {
        	s += three(mode, rIn, beta) + " " + pThree(mode, power, beta) + " ";
        }
        s += formPwr(oPower, 'watts');
        if (scale !== 'watts') { s += "  " + formPwr(oPower, scale); }
        s += eol;
    }
    return s;
}
