/*
	Electrical Networks - Electrical network calculations
	Copyright © 2013 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA 02110-1301	USA

	Electrical Networks - version 2.1
	27 February, 2013
	Copyright 2013 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*properties
	BTpad, Bpad, Delta, LN10, LOG10E, Latten, PIatten, Star, Tatten, attenB, 
	attenBT, attenL, attenLL, attenLS, attenPI, attenT, exp, floor, gain, log, 
	p10, p12, p20, p23, p30, p31, p34, p41, pIn, pOut, prototype, r0, r10, r12, 
	r20, r23, r30, r31, r34, r41, rIn, rOut, setPrecision, sqrt, toDelta, 
	toFixed, toStar, toString, zIn, zOut
*/

/*
	NETWORK is a global object whose properties are functions.
	
	Notation:
		rIn		source impedance
		rOut	load impedance
		r0		source and load impedance
		gain	required (or attained) power gain as a ratio (0 <= gain <= 1)
				or as a number of decibels (gain < 0)
		rXY		resistor between terminal X and terminal Y
		zIn		computed input impedance of the network
		zOut	computed output impedance of the network
		pIn		power from source into network
		pXY		power dissipated in resistor rXX
		pOut	power into load

	NETWORK.Star
		Constructor
			new NETWORK.Star(r10, r20, r30)		// makes a Star object
		Methods
			toString()
			toDelta()
		
	NETWORK.Delta
		Constructor
			new NETWORK.Delta(r12, r23, r31)	// makes a Delta object
		Methods
			toString()
			toStar()

	NETWORK.Tatten
		Constructor
			new NETWORK.Tatten(rIn, rOut, gain, r10, r20, r30, zIn, zOut, pIn, p10, p20, p30, pOut)
		Methods
			toString()
	
	NETWORK.attenT(rIn, rOut, gain, pIn)		// design T-network attenuator
												// returns a NETWORK.Tatten object
	
	NETWORK.PIatten
		Constructor
			new NETWORK.PIatten(rIn, rOut, gain, r12, r23, r31, zIn, zOut, pIn, p12, p23, p31, pOut)
		Methods
			toString()
			
	NETWORK.attenPI(rIn, rOut, gain, pIn)		// design PI-network attenuator
												// returns a NETWORK.attenPI object
	
	NETWORK.Latten
		Constructor
			new NETWORK.Latten(rIn, rOut, gain, r10, r20, r30, zIn, zOut, pIn, p10, p20, p30, pOut)
		Methods
			toString()

	NETWORK.attenL(rIn, rOut, pIn)				// design L-network attenuator
												// returns a NETWORK.Latten object
												
	NETWORK.attenLS(rIn, rOut, gain, pIn)		// design LS-network attenuator
												// returns a NETWORK.Latten object
												
	NETWORK.attenLL(rIn, rOut, gain, pIn)		// design LL-network attenuator
												// returns a NETWORK.Latten object

	NETWORK.BTpad
		Constructor
			new NETWORK.BTpad(r0, gain, r12, r10, r20, r30, pIn, p12, p10, p20, p30, pOut)
		Method			
			toString()
	
	NETWORK.attenBT(r0, gain, pIn)				// design BT-network attenuator
												// returns a NETWORK.BTpad object

	NETWORK.Bpad
		Constructor
			new NETWORK.Bpad(r0, gain, r12, r23, r34, r41, pIn, p12, p23, p34, p41, pOut)
		Methods
			toString()
			
	NETWORK.attenB(r0, gain, pIn)				// design B-network attenuator
												// returns a NETWORK.Bpad object
												
	NETWORK.setPrecision(p) // sets number of decimal places used in toString() methods.					
*/

var NETWORK = (function () {
	var itself = {},
		prec = 3;

	function log10(x) {
		return Math.LOG10E * Math.log(x);
	}

	function exp10(x) {
		return Math.exp(Math.LN10 * x);
	}

	function ratio(gain) {
		if (gain < 0) { // gain in dB
			return exp10(gain / 10); // power ratio
		}
		if (gain <= 1) {
			return gain;			// power ratio
		}
		throw new Error("Invalid gain value: " + gain.toFixed(prec));
	}

	function parallel(r1, r2) {
		if (r1 === Infinity) { return r2; }
		if (r2 === Infinity) { return r1; }
		if (r1 === 0) { return 0; }
		if (r2 === 0) { return 0; }
		return r1 * r2 / (r1 + r2);
	}

	itself.Star = function (r10, r20, r30) {
		this.r10 = r10;
		this.r20 = r20;
		this.r30 = r30;
	};

	itself.Delta = function (r12, r23, r31) {
		this.r12 = r12;
		this.r23 = r23;
		this.r31 = r31;
	};

	itself.Star.prototype.toDelta = function () {
		var r12 = this.r10 * this.r20 / this.r30 + this.r10 + this.r20,
			r23 = this.r20 * this.r30 / this.r10 + this.r20 + this.r30,
			r31 = this.r30 * this.r10 / this.r20 + this.r30 + this.r10;

		return new itself.Delta(r12, r23, r31);
	};

	itself.Delta.prototype.toStar = function () {
		var den = this.r12 + this.r23 + this.r31,
			r10 = this.r31 * this.r12 / den,
			r20 = this.r12 * this.r23 / den,
			r30 = this.r23 * this.r31 / den;

		return new itself.Star(r10, r20, r30);
	};

	itself.Star.prototype.toString = function () {
		return "Y["
			+ this.r10.toFixed(prec) + ", "
			+ this.r20.toFixed(prec) + ", "
			+ this.r30.toFixed(prec) + "]";
	};

	itself.Delta.prototype.toString = function () {
		return "D["
			+ this.r12.toFixed(prec) + ", "
			+ this.r23.toFixed(prec) + ", "
			+ this.r31.toFixed(prec) + "]";
	};

	itself.Tatten = function (rIn, rOut, gain, r10, r20, r30, zIn, zOut, pIn, p10, p20, p30, pOut) { // T-network
		this.rIn  = rIn;
		this.rOut = rOut;
		this.gain = gain;

		this.r10  = r10;
		this.r20  = r20;
		this.r30  = r30;
		
		this.zIn  = zIn;
		this.zOut = zOut;
		
		this.pIn  = pIn;
		this.p10  = p10;
		this.p20  = p20;
		this.p30  = p30;
		this.pOut = pOut;
	};

	itself.Tatten.prototype.toString = function () {
		return "T["
			+ this.rIn.toFixed(prec) + ", "
			+ this.rOut.toFixed(prec) + ", "
			+ this.gain.toFixed(prec) + ", "

			+ this.r10.toFixed(prec) + ", "
			+ this.r20.toFixed(prec) + ", "
			+ this.r30.toFixed(prec) + ", "
			
			+ this.zIn.toFixed(prec) + ", "
			+ this.zOut.toFixed(prec) + ", "

			+ this.pIn.toFixed(prec) + ", "
			+ this.p10.toFixed(prec) + ", "
			+ this.p20.toFixed(prec) + ", "
			+ this.p30.toFixed(prec) + ", "
			+ this.pOut.toFixed(prec) + "]";
	};

	itself.attenT = function (rIn, rOut, gain, pIn) {
		var beta2 = ratio(gain),
			beta = Math.sqrt(beta2),
			tmp = (1 + beta2) / (2 * beta),
			tmp2 = rIn >= rOut ? rIn / rOut : rOut / rIn,
			nok = tmp2 === tmp * tmp,
			ok = tmp2 < tmp * tmp,
			r10,
			r20,
			r30,
			zIn,
			zOut,
			p10,
			p20,
			p30,
			pOut;

		if (nok) { return new itself.Tatten(rIn, rOut, gain, 0, 0, Infinity, rIn, rOut, pIn, 0, 0, 0, pIn); }

		if (ok) {
			r30 = Math.sqrt(rIn * rOut) * 2 * beta / (1 - beta2);
			r10 = rIn * (1 + beta2) / (1 - beta2) - r30;
			r20 = rOut * (1 + beta2) / (1 - beta2) - r30;

			zIn = r10 + parallel(r30, r20 + rOut);
			zOut = r20 + parallel(r30, r10 + rIn);

			pOut = pIn * beta2;
			p10 = pIn * r10 / rIn;
			p20 = pIn * r20 * beta2 / rOut;
			p30 = pIn - p10 - p20 - pOut;

			return new itself.Tatten(rIn, rOut, gain, r10, r20, r30, zIn, zOut, pIn, p10, p20, p30, pOut);
		}

		tmp = Math.sqrt(tmp2) - Math.sqrt(tmp2 - 1);
		return new itself.Tatten(rIn, rOut, gain, NaN, 20 * log10(tmp), tmp * tmp, NaN, NaN, NaN, NaN, NaN, NaN, NaN);
	};

	itself.PIatten = function (rIn, rOut, gain, r12, r23, r31, zIn, zOut, pIn, p12, p23, p31, pOut) { // PI-network
		this.rIn  = rIn;
		this.rOut = rOut;
		this.gain = gain;

		this.r12  = r12;
		this.r23  = r23;
		this.r31  = r31;
		
		this.zIn  = zIn;
		this.zOut = zOut;
		
		this.pIn  = pIn;
		this.p12  = p12;
		this.p23  = p23;
		this.p31  = p31;
		this.pOut = pOut;
	};

	itself.PIatten.prototype.toString = function () {
		return "PI["
			+ this.rIn.toFixed(prec) + ", "
			+ this.rOut.toFixed(prec) + ", "
			+ this.gain.toFixed(prec) + ", "

			+ this.r12.toFixed(prec) + ", "
			+ this.r23.toFixed(prec) + ", "
			+ this.r31.toFixed(prec) + ", "
			
			+ this.zIn.toFixed(prec) + ", "
			+ this.zOut.toFixed(prec) + ", "

			+ this.pIn.toFixed(prec) + ", "
			+ this.p12.toFixed(prec) + ", "
			+ this.p23.toFixed(prec) + ", "
			+ this.p31.toFixed(prec) + ", "
			+ this.pOut.toFixed(prec) + "]";
	};

	itself.attenPI = function (rIn, rOut, gain, pIn) {
		var beta2 = ratio(gain),
			beta = Math.sqrt(beta2),
			tmp = (1 + beta2) / (2 * beta),
			tmp2 = rIn >= rOut ? rIn / rOut : rOut / rIn,
			nok = tmp2 === tmp * tmp,
			ok = tmp2 < tmp * tmp,
			r12,
			r23,
			r31,
			zIn,
			zOut,
			p12,
			p23,
			p31,
			pOut;

		if (nok) { return new itself.PIatten(rIn, rOut, gain, 0, Infinity, Infinity, rIn, rOut, pIn, 0, 0, 0, pIn); }

		if (ok) {
			r12 = Math.sqrt(rIn * rOut) * (1 - beta2) / (2 * beta);
			r23 = rOut * (1 - beta2) / (1 + beta2 - 2 * beta * Math.sqrt(rOut / rIn));
			r31 = rIn * (1 - beta2) / (1 + beta2 - 2 * beta * Math.sqrt(rIn / rOut));
			
			zIn = parallel(r31, r12 + parallel(r23, rOut));
			zOut = parallel(r23, r12 + parallel(r31, rIn));

			pOut = pIn * beta2;
			p23 = pIn * rOut * beta2 / r23;
			p31 = pIn * rIn / r31;
			p12 = pIn - p23 - p31 - pOut;
			return new itself.PIatten(rIn, rOut, gain, r12, r23, r31, zIn, zOut, pIn, p12, p23, p31, pOut);
		}

		tmp = Math.sqrt(tmp2) - Math.sqrt(tmp2 - 1);
		return new itself.PIatten(rIn, rOut, gain, NaN, 20 * log10(tmp), tmp * tmp, NaN, NaN, NaN, NaN, NaN, NaN, NaN);
	};

	itself.Latten = function (rIn, rOut, gain, r10, r20, r30, zIn, zOut, pIn, p10, p20, p30, pOut) { // L-network
		this.rIn  = rIn;
		this.rOut = rOut;
		this.gain = gain;

		this.r10  = r10;
		this.r20  = r20;
		this.r30  = r30;

		this.zIn  = zIn;
		this.zOut = zOut;
		
		this.pIn  = pIn;
		this.p10  = p10;
		this.p20  = p20;
		this.p30  = p30;
		this.pOut = pOut;
	};

	itself.Latten.prototype.toString = function () {
		return "L["
			+ this.rIn.toFixed(prec) + ", "
			+ this.rOut.toFixed(prec) + ", "
			+ this.gain.toFixed(prec) + ", "

			+ this.r10.toFixed(prec) + ", "
			+ this.r20.toFixed(prec) + ", "
			+ this.r30.toFixed(prec) + ", "
 
			+ this.zIn.toFixed(prec) + ", "
			+ this.zOut.toFixed(prec) + ", "

			+ this.pIn.toFixed(prec) + ", "
			+ this.p10.toFixed(prec) + ", "
			+ this.p20.toFixed(prec) + ", "
			+ this.p30.toFixed(prec) + ", "
			+ this.pOut.toFixed(prec) + "]";
	};

	itself.attenL = function (rIn, rOut, pIn) {
		var r10, r20, r30, beta, gain, zIn, zOut, p10, p20, p30, pOut;
 
		if (rIn > rOut) {
			r10 = Math.sqrt(rIn * (rIn - rOut));
			r20 = 0;
			r30 = rOut * Math.sqrt(rIn / (rIn - rOut));
			beta = 1 / (Math.sqrt(rIn / rOut - 1) + Math.sqrt(rIn / rOut));
			gain = beta * beta;

			zIn = r10 + parallel(r30, rOut);
			zOut = parallel(r30, r10 + rIn);
			
			pOut = pIn * beta * beta;
			p10 = pIn * r10 / rIn;
			p20 = 0;
			p30 = pIn - p10 - pOut;
		} else if (rOut > rIn) {
			r10 = 0;
			r20 = Math.sqrt(rOut * (rOut - rIn));
			r30 = rIn * Math.sqrt(rOut / (rOut - rIn));
			beta = 1 / (Math.sqrt(rOut / rIn - 1) + Math.sqrt(rOut / rIn));
			gain = beta * beta;

			zIn = parallel(r30, r20 + rOut);
			zOut = r20 + parallel(r30, rIn);
			
			pOut = pIn * beta * beta;
			p10 = 0;
			p20 = pOut * r20 / rOut;
			p30 = pIn - p20 - pOut;
		} else {
			r10 = 0;
			r20 = 0;
			r30 = Infinity;
			beta = 1;
			gain = 1;

			zIn = rOut;
			zOut = rIn;

			p10 = 0;
			p20 = 0;
			p30 = 0;
			pOut = pIn;
		}

		return new itself.Latten(rIn, rOut, gain, r10, r20, r30, zIn, zOut, pIn, p10, p20, p30, pOut);
	};

	itself.attenLS = function (rIn, rOut, gain, pIn) {
		var beta2 = ratio(gain),
			beta = Math.sqrt(beta2),
			s2 = rIn / rOut,
			s = Math.sqrt(s2),
			ok = (beta <= s)  && (beta <= 1 / s),
			r10,
			r20,
			r30,
			zIn,
			zOut,
			p10,
			p20,
			p30,
			pOut,
			tmp;

		if (ok) {
			if (rIn >= rOut) {
				r10 = rIn * (s - beta) / s;
				r20 = 0;
				r30 = rIn * beta / (s - beta * s2);
				
				zIn = r10 + parallel(r30, rOut);
				zOut = parallel(rIn + r10, r30);
				
				if (r30 === Infinity) {
					tmp = 1;
				} else {
					tmp = r30 / (r30 + rOut);
				}
				gain = tmp * tmp * rOut / zIn;
				
				if (r30 === Infinity) {
					tmp = rOut / (r10 + rOut);
				} else {
					tmp = r30 * rOut / (r30 + rOut);
					tmp = tmp / (r10 + tmp);
				}

				if (r10 === 0) {
					p10 = 0;
				} else {
					p10 = pIn * (1 - tmp) * (1 - tmp) * rIn / r10;
				}
				p20 = 0;
				if (r30 === 0) {
					p30 = 0;
				} else {
					p30 = pIn * tmp * tmp * rIn / r30;
				}
				pOut = pIn * beta2;
			} else {
				r10 = 0;
				r20 = rIn * (s - beta) / (beta * s2);
				r30 = rIn / (1 - beta * s);

				zIn = parallel(r30, r20 + rOut);
				zOut = r20 + parallel(rIn, r30);
				
				tmp = rOut / (r20 + rOut);
				gain = tmp * tmp * zIn / rOut;
				
				p10 = 0;
				p30 = pIn * rIn / r30;
				pOut = pIn * beta2;
				p20 = pIn - pOut - p30;
			}
			return new itself.Latten(rIn, rOut, gain, r10, r20, r30, zIn, zOut, pIn, p10, p20, p30, pOut);
		}

		return new itself.Latten(rIn, rOut, 20 * log10(beta), NaN, s >= 1 ? -20 * log10(s) : 20 * log10(s), s >= 1 ? 1 / s2 : s2, NaN, NaN, NaN, NaN, NaN, NaN, NaN);
	};

	itself.attenLL = function (rIn, rOut, gain, pIn) {	 // experimental
		var beta2 = ratio(gain),
			beta = Math.sqrt(beta2),
			s2 = rIn / rOut,
			s = Math.sqrt(s2),
			ok = (beta <= s)  && (beta <= 1 / s),
			r10,
			r20,
			r30,
			zIn,
			zOut,
			tmp,

			p10,
			p20,
			p30,
			pOut;
			
		if (ok) {
			if (rIn >= rOut) {
				r10 = rIn * (1 - beta * s) / (beta * s);
				r20 = 0;
				r30 = rIn / (s2 - beta * s);

				zIn = r10 + parallel(r30, rOut);
				zOut = parallel(rIn + r10, r30);
				
				if (r30 === Infinity) {
					tmp = 1;
				} else {
					tmp = r30 / (r30 + rOut);
				}
				gain = tmp * tmp * rOut / zIn;
			   
				pOut = pIn * tmp * tmp * rOut / zIn;
				p20 = 0;
				p30 = pOut * rOut / r30;
				p10 = pIn - p30 - pOut;
			} else {
				r10 = 0;
				r20 = rIn * (1 - beta * s) / s2;
				r30 = rIn * beta / (s - beta);
				
				zIn = parallel(r30, r20 + rOut);
				zOut = r20 + parallel(rIn, r30);
				
				tmp = rOut / (r20 + rOut);
				gain = tmp * tmp * zIn / rOut;

				pOut = pIn * tmp * tmp * zIn / rOut;
				p10 = 0;
				p30 = pIn * zIn / r30;
				p20 = pIn - pOut - p30;
			}
			
			return new itself.Latten(rIn, rOut, gain, r10, r20, r30, zIn, zOut, pIn, p10, p20, p30, pOut);
		}

		return new itself.Latten(rIn, rOut, 20 * log10(beta), NaN, s >= 1 ? -20 * log10(s) : 20 * log10(s), s >= 1 ? 1 / s2 : s2, NaN, NaN, NaN, NaN, NaN, NaN, NaN);
	};

	itself.BTpad = function (r0, gain, r12, r10, r20, r30, pIn, p12, p10, p20, p30, pOut) {
		this.r0 = r0;
		this.gain = gain;
		
		this.r12 = r12;
		this.r10 = r10;
		this.r20 = r20;
		this.r30 = r30;

		this.pIn = pIn;
		this.p12 = p12;
		this.p10 = p10;
		this.p20 = p20;
		this.p30 = p30;
		this.pOut = pOut;
	};

	itself.BTpad.prototype.toString = function () {
		return "BT["
			+ this.r0.toFixed(prec)	  + ", "
			+ this.gain.toFixed(prec) + ", "

			+ this.r12.toFixed(prec) + ", "
			+ this.r10.toFixed(prec) + ", "
			+ this.r20.toFixed(prec) + ", "
			+ this.r30.toFixed(prec) + ", "

			+ this.pIn.toFixed(prec) + ", "
			+ this.p12.toFixed(prec) + ", "
			+ this.p10.toFixed(prec) + ", "
			+ this.p20.toFixed(prec) + ", "
			+ this.p30.toFixed(prec) + ", "
			+ this.pOut.toFixed(prec) + "]";
	};

	itself.attenBT = function (r0, gain, pIn) {
		var beta2 = ratio(gain),
			beta = Math.sqrt(beta2),
			r12 = r0 * (1 - beta) / beta,
			r10 = r0,
			r20 = r0,
			r30 = r0 * beta / (1 - beta),

			p12 = pIn * beta * (1 - beta),
			p10 = pIn * (1 - beta) * (1 - beta),
			p20 = 0,
			p30 = pIn * beta * (1 - beta),
			pOut = pIn * beta2;

		return new itself.BTpad(r0, gain, r12, r10, r20, r30, pIn, p12, p10, p20, p30, pOut);
	};

	itself.Bpad = function (r0, gain, r12, r23, r34, r41, pIn, p12, p23, p34, p41, pOut) {
		this.r0 = r0;
		this.gain = gain;

		this.r12 = r12;
		this.r23 = r23;
		this.r34 = r34;
		this.r41 = r41;

		this.pIn = pIn;
		this.p12 = p12;
		this.p23 = p23;
		this.p34 = p34;
		this.p41 = p41;
		this.pOut = pOut;
	};

	itself.Bpad.prototype.toString = function () {
		return "B["
			+ this.r0.toFixed(prec)	  + ", "
			+ this.gain.toFixed(prec) + ", "

			+ this.r12.toFixed(prec) + ", "
			+ this.r23.toFixed(prec) + ", "
			+ this.r34.toFixed(prec) + ", "
			+ this.r41.toFixed(prec) + ", "
			
			+ this.pIn.toFixed(prec) + ", "
			+ this.p12.toFixed(prec) + ", "
			+ this.p23.toFixed(prec) + ", "
			+ this.p34.toFixed(prec) + ", "
			+ this.p41.toFixed(prec) + ", "
			+ this.pOut.toFixed(prec) + "]";
	};

	itself.attenB = function (r0, gain, pIn) {
		var beta2 = ratio(gain),
			beta = Math.sqrt(beta2),
			r12 = r0 * (1 - beta) / (1 + beta),
			r23 = r0 * (1 + beta) / (1 - beta),
			r34 = r12,
			r41 = r23,

			p12 = pIn * (1 - beta2) / 4,
			p23 = p12,
			p34 = p12,
			p41 = p12,
			pOut = pIn * beta2;

		return new itself.Bpad(r0, gain, r12, r23, r34, r41, pIn, p12, p23, p34, p41, pOut);
	};

	itself.setPrecision = function (p) {
		if ((typeof p === "number") && (p >= 0) && (p < 20)) {
			prec = Math.floor(p);
		}
	};

	return itself;
}());

/*
// Test program follows

var star, delta, attenT, attenPI, attenT2, attenPI2,
	attenL, attenLS, attenLL, attenBT, attenB, i;

NETWORK.setPrecision(5);

print("	 ");
print("--");

star = new NETWORK.Star(199.4, 199.4, 803.2); // define star
print("Star:								  " + star);
delta = star.toDelta();		// transform to delta
print("Delta::								  " + delta);
star = delta.toStar();		// transform back again
print("Star::								  " + star);

print("--");

attenT = NETWORK.attenT(600, 600, 0.25, 100);
print("attenT(600, 600, 0.25, 100):			  " + attenT);
attenT = NETWORK.attenT(600, 600, -6.0206, 100);
print("attenT(600, 600, -6.0206, 100):		  " + attenT);
attenT = NETWORK.attenT(600, 600, -6, 100);
print("attenT(600, 600, -6, 100):			  " + attenT);

print("--");

attenPI = NETWORK.attenPI(600, 600, -6, 100);
print("attenPI(600, 600, -6, 100):			  " + attenPI);

print("--");

attenT2 = NETWORK.attenT(400, 70, -13, 100); // fails
print("attenT(400, 70, -13, 100):			  " + attenT2);
if (isNaN(attenT2.r10)) {
	print("Minimum attenuation is				  " + -attenT2.r20.toFixed(5) + " dB");
}
attenT2 = NETWORK.attenT(400, 70, -13.18254, 100);
print("attenT(400, 70, -13.18254, 100):		  " + attenT2);
attenPI2 = NETWORK.attenPI(400, 70, -13, 100); // fails
print("attenPI(400, 70, -13, 100):			  " + attenPI2);
if (isNaN(attenPI2.r12)) {
	print("Minimum attenuation is				  " + -attenPI2.r23.toFixed(5) + " dB");
}
attenPI2 = NETWORK.attenPI(400, 70, -13.18254, 100);
print("attenPI(400, 70, -13.18254, 100):	  " + attenPI2);

print("--");

attenL = NETWORK.attenL(400, 70, 100);
print("attenL(400, 70, 100):				  " + attenL);
print("--");
attenL = NETWORK.attenL(70, 400, 100);
print("attenL(70, 400, 100):				  " + attenL);
print("--");
attenL = NETWORK.attenL(300, 75, 100);
print("attenL(300, 75, 100):				  " + attenL);
print("--");
attenL = NETWORK.attenL(75, 300, 100);
print("attenL(75, 300, 100):				  " + attenL);

print("--");

attenLS = NETWORK.attenLS(8, 8, 0.25, 100);
print("attenLS(8, 8, 0.25, 100):			  " + attenLS);
print("--");
attenLL = NETWORK.attenLL(8, 8, 0.25, 100);
print("attenLL(8, 8, 0.25, 100):			  " + attenLL);
print("--");
print("	 ");
attenLS = NETWORK.attenLS(75, 50, -12, 100);
print("attenLS((75, 50, -12, 100):			  " + attenLS);
print("--");
attenLL = NETWORK.attenLL(75, 50, -12, 100);
print("attenLL(75, 50, -12, 100):			  " + attenLL);
print("--");
print("	 ");
attenLS = NETWORK.attenLS(50, 75, -6, 100);
print("attenLS(50, 75, -6, 100):			  " + attenLS);
print("--");
attenLL = NETWORK.attenLL(50, 75, -6, 100);
print("attenLL(50, 75, -6, 100):			  " + attenLL);
print("--");
print("	 ");
attenLS = NETWORK.attenLS(50, 75, -1, 100);
print("attenLS(50, 75, -1, 100):			  " + attenLS);
print("--");
attenLS = NETWORK.attenLS(50, 300, 0.25, 100);
print("attenLS(50, 300, 0.25, 100):			  " + attenLS);
print("--");
attenLS = NETWORK.attenLS(300, 75, 0.3, 100);
print("attenLS(300, 75, 0.3, 100):			  " + attenLS);
attenLS = NETWORK.attenLS(300, 75, 0.26, 100);
print("attenLS(300, 75, 0.26, 100):			  " + attenLS);
attenLS = NETWORK.attenLS(300, 75, 0.25, 100);
print("attenLS(300, 75, 0.25, 100):			  " + attenLS);
attenLS = NETWORK.attenLS(300, 75, 0.24, 100);
print("attenLS(300, 75, 0.24, 100):			  " + attenLS);
attenLS = NETWORK.attenLS(300, 75, 0.2, 100);
print("attenLS(300, 75, 0.2, 100):			  " + attenLS);
print("	 ");
print("--");

for (i = 20; i < 31; i += 1) {
	attenLS = NETWORK.attenLS(300, 75, i / 100, 100);
	print("attenLS(300, 75, " + (i / 100).toFixed(2) + ", 100):			  " + attenLS);
}
print("	 ");
print("--");
for (i = 20; i < 31; i += 1) {
	attenLS = NETWORK.attenLS(75, 300, i / 100, 100);
	print("attenLS(75, 300, " + (i / 100).toFixed(2) + ", 100):			  " + attenLS);
}
print("	 ");
print("--");
for (i = 20; i < 31; i += 1) {
	attenLL = NETWORK.attenLL(300, 75, i / 100, 100);
	print("attenLL(300, 75, " + (i / 100).toFixed(2) + ", 100):			  " + attenLL);
}
print("	 ");
print("--");
for (i = 20; i < 31; i += 1) {
	attenLL = NETWORK.attenLL(75, 300, i / 100, 100);
	print("attenLL(75, 300, " + (i / 100).toFixed(2) + ", 100):			  " + attenLL);
}

print("--");

attenBT = NETWORK.attenBT(600, 0.5, 100);
print("attenBT(600, 0.5, 100):				  " + attenBT);
attenBT = NETWORK.attenBT(600, -3.0103, 100);
print("attenBT((600, -3.0103, 100):			  " + attenBT);
attenBT = NETWORK.attenBT(600, -3, 100);
print("attenBT(600, -3, 100):				  " + attenBT);
attenBT = NETWORK.attenBT(600, 0.25, 100);
print("attenBT(600, 0.25, 100):				  " + attenBT);
attenBT = NETWORK.attenBT(600, -6.0206, 100);
print("attenBT(600, -6.0206, 100):			  " + attenBT);
attenBT = NETWORK.attenBT(600, -6, 100);
print("attenBT(600, -6, 100):				  " + attenBT);
attenBT = NETWORK.attenBT(600, 0.1, 100);
print("attenBT(600, 0.1, 100):				  " + attenBT);
attenBT = NETWORK.attenBT(600, -10, 100);
print("attenBT(600, -10, 100):				  " + attenBT);
attenBT = NETWORK.attenBT(650, -50, 100);
print("attenBT(650, -50, 100):				  " + attenBT);
attenBT = NETWORK.attenBT(600, -20, 100);
print("attenBT(600, -20, 100):				  " + attenBT);

print("--");

for (i = 1; i < 10; i += 1) {
	attenB = NETWORK.attenB(1, -i, 100);
	print("attenB(1, -" + i + ", 100):					  " + attenB);
}

print("--");

attenB = NETWORK.attenB(1, -3.0103, 100);
print("attenB(1, -3.0103, 100):				  " + attenB);
attenB = NETWORK.attenB(1, -6.0206, 100);
print("attenB(1, -6.0206, 100):				  " + attenB);
attenB = NETWORK.attenB(600, -40, 100);
print("attenB(600, -40, 100):				  " + attenB);

print("--");

for (i = 10; i < 100; i += 10) {
	attenB = NETWORK.attenB(1, -i, 100);
	print("attenB(1, -" + i + ", 100):					 " + attenB);
}
*/
