/*
	Beat Meter - A beat meter and metronome.
	Copyright © 2005-2019 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA 02110-1301	USA

	Beat Meter - version 2.1 - browser version
	30 April, 2019
	Copyright © 2005-2019 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*jslint bitwise, browser, this */

/*global drawImage, newImage, moveObj, newCanvas, newCanvasImage, newSelector, newText,
	window, scale, playSound, a440Buffer, clickBuffer, setPreference, addToMenu,
	newInput, requestAnimationFrame, unlockSound, initSound
*/

/*property
    PI, altKey, checked, clientX, clientY, color, cos, forEach, innerHTML,
    isNaN, now, onchange, onclick, onload, onmousedown, onmousemove, onmouseup,
    ontouchend, ontouchmove, ontouchstart, opacity, open, preventDefault, round,
    selectedIndex, setAttribute, style, targetTouches, title, vRegP, value
*/

//////////////////////////////////// Start of the gui ////////////////////////////////////

// newImage(hOffset, vOffset, width, height, src, zOrder, opacity)
// newText(hOffset, vOffset, width, height, value, zOrder, style)
// newCanvas(hOffset, vOffset, width, height, src, zOrder, opacity, hRegP, vRegP)

var base = "Resources/Images/";

var button = newImage(0, 0, 512, 768, base + "button.png", 1, 1.0);
button.title = (
	"Click the upper area to toggle the metronome on or off.\n" +
	"Click the lower area repeatedly in time with a beat to measure the beat rate (bpm).\n" +
	"Alt-click the lower area to copy a measured beat rate to the metronome."
);

var rates = newImage(216, 132, 70, 316, base + "rates.png", 2, 1.0);

var hRegP = 456;
var vRegP = 456;

var pendulum = newCanvas(251 - hRegP, 540 - vRegP, 912, 912, base + "pendulum.png", 3, 1.0, hRegP, vRegP);	//was 456
function pendulumLoaded(img) {
	drawImage(pendulum, img, 0, 0, 0);
}
var pendulumImage = newCanvasImage(pendulum, pendulumLoaded);

vRegP = 406;	//for BPM 92

var bob = newCanvas(251 - 456, 540 - vRegP, 912, 912, base + "bob.png", 4, 1.0, 456, vRegP);
function bobLoaded(img) {
	drawImage(bob, img, 0, 0, 0);
}
var bobImage = newCanvasImage(bob, bobLoaded);
bob.title = (
	"Drag the pendulum bob up or down to alter the rate of the metronome.\n" +
	"Alt-click the pendulum bob to tune up (A440)."
);

var cover = newImage(0, 506, 512, 262, base + "cover.png", 5, 1.0);
cover.title = (
	"Click the upper area to toggle the metronome on or off.\n" +
	"Click the lower area repeatedly in time with a beat to measure the beat rate (bpm).\n" +
	"Alt-click the lower area to copy a measured beat rate to the metronome."
);

var buttons = newImage(0, 436, 510, 68, base + "buttons.png", 6, 1.0);

var style = "font-family:LucidaGrande;font-size:48px;color:#FFFF00;background-color:transparent;border:solid 1px white;text-align:center";
//var style = "font-family:Digital-7;font-size:48px;color:#FFFF00;background-color:transparent;border:none;text-align:center";
var beatsPerMin = newText(188, 530, 128, 60, "92", 6, style);

style = "font-family:LucidaGrande;font-size:48px;color:#FFFF00;background-color:transparent;border:none;text-align:center";
var diags = newText(188, 615, 128, 60, "", 6, style);
diags.style.opacity = "1.0";
//diags.style.border = "none";

var helpButton = newImage(390, 0, 30, 22, base + "help.png", 2);   // customized
helpButton.title = "Displays information about this program.";

helpButton.onmousedown = function () {
	"use strict";
	this.style.opacity = "0.5";
};
helpButton.onmouseup = function () {
	"use strict";
	this.style.opacity = "1.0";
	window.open("Help.html");
};

style = "font-family:LucidaGrande;font-size:14px;color:black;background-color:white;border:solid 1px black;text-align:center";
var bpmInput = newInput(85, 0, 25, 20, 92, 2, style);
bpmInput.title = "To alter the rate of the metronome, enter an integer between 40 and 208, then press RETURN.";

///////////////////////////////////// Preference Code ////////////////////////////////////

var beatRatePref;
var sampleLengthPref;
var metroScalePref;
var diagControlPref;

//////////////////////////////////////// Alt-Key /////////////////////////////////////////

var altKey = newInput(237, 1, 8, 16, "alt-key", 3);
altKey.setAttribute("type", "checkbox");
altKey.title = "Enable/disable the on-screen alt-key.";

var altButton = newInput(175, 0, 8, 16, "alt-key", 3);
altButton.setAttribute("type", "button");

altButton.onclick = function () {
	"use strict";
	altKey.checked = !altKey.checked;
};
altButton.title = "Enable/disable the on-screen alt-key.";


///////////////////////////////////// Diags Control //////////////////////////////////////

var diagControl = newInput(260, 1, 8, 16, "diagControl", 2);
diagControl.setAttribute("type", "checkbox");
diagControl.title = "Check this box to enable diagnostics.";
diagControl.checked = (diagControlPref === "1");

diagControl.onchange = function () {
	"use strict";
	diagControlPref = (
		diagControl.checked
		? "1"
		: "0"
	);
	setPreference("diagControlPref", diagControlPref);
	if (!diagControl.checked) {
		diags.innerHTML = "";
	}
};

//////////////////////////////////// Menus Customized ////////////////////////////////////

var MenuArray = [
	["5", "10", "15", "20"]
];

var MenuTitleArray = [
	"Select the number of samples over which the running average is to be computed."
];

var MenuSelectedIndex = [1];

var Menu = [];

Menu[0] = newSelector(320, 0, 110, 20, "", 2);
//Menu[1] = newSelector(310, 17, 56, 20, "", 2);
//Menu[2] = newSelector(210, 80, 110, 20, "", 2);
//Menu[3] = newSelector(340, 80, 56, 20, "", 2);

(function () {
	"use strict";
	MenuArray.forEach(function (ele, i) {
		addToMenu(Menu[i], ele);
		Menu[i].selectedIndex = MenuSelectedIndex[i];
		Menu[i].title = MenuTitleArray[i];
	});
}());

var sampleLengthMenu = Menu[0];

//////////////////////////////////////////////////////////////////////////////////////////

var metroScale = newInput(280, 1, 8, 16, "metroScale", 2);
metroScale.setAttribute("type", "checkbox");
metroScale.title = "Check this box to show the metronome scale.";
metroScale.checked = (metroScalePref === "1");

rates.style.opacity = (
	metroScalePref === "1"
	? 1.0
	: 0.0
);


metroScale.onchange = function () {
	"use strict";
	metroScalePref = (
		metroScale.checked
		? "1"
		: "0"
	);
	setPreference("metroScalePref", metroScalePref);
	rates.style.opacity = (
		metroScalePref === "1"
		? 1.0
		: 0.0
	);
};

//////////////////////////////////////////////////////////////////////////////////////////

function bpm(val) { // 300 <= val <= 500
	"use strict";
	if (val <= 300) {
		return Math.round(208 - 24 * (val - 200) / 43);
	}
	if (val <= 360) {
		return Math.round(144 - 56 * (val - 314) / 100);
	}
	if (val <= 442) {
		return Math.round(116 - 44 * (val - 364) / 78);
	}
	if (val <= 454) {
		return Math.round(69 - 3 * (val - 448) / 6);
	}
	return Math.round(63 - 23 * (val - 459) / 41);
}

function val(bpm) { // 40 <= val <= 208
	"use strict";
	if (bpm <= 60) {
		return Math.round(500 - 37 * (bpm - 40) / 20);
	}
	if (bpm <= 72) {
		return Math.round(459 - 17 * (bpm - 63) / 9);
	}
	if (bpm <= 120) {
		return Math.round(435 - 78 * (bpm - 76) / 44);
	}
	if (bpm <= 144) {
		return Math.round(346 - 32 * (bpm - 126) / 18);
	}
	return Math.round(300 - 100 * (bpm - 152) / 56);
}

//////////////////////////////////////////////////////////////////////////////////////////

/*
function screen_X() {
	"use strict";
	if (window.screenX !== undefined) {
		return window.screenX;
	}
	return window.screenLeft;
}

function screen_Y() {
	"use strict";
	if (window.screenY !== undefined) {
		return window.screenY;
	}
	return window.screenTop;
}
*/

//////////////////////////////////////////////////////////////////////////////////////////

var metroBeatRate;
var moveEnabled = false;
var maxRotation = 36;
var intervalID = null;
var runEnabled = false;

metroBeatRate = Number(beatRatePref);
beatsPerMin.innerHTML = beatRatePref;
bpmInput.value = beatRatePref;

bob.vRegP = Math.round(val(metroBeatRate) * scale);
beatsPerMin.style.color = "#FFFF00";
moveObj(bob, 251 - 456, 540 - bob.vRegP);
drawImage(bob, bobImage, 0, 0, 0);

var run = function () {
	"use strict";
	var tau = 120 / metroBeatRate;					// s		period of the pendulum in seconds
	var omega = 2 * Math.PI / tau;
	var angle = maxRotation;						// deg		initial angle of the pendulum from the vertical in degrees
	var lastAngle = angle;
	var oldTime = Date.now();
	var time;
	var t;
	var count = 0;

	function draw() {
		if (intervalID === null) {
			return;
		}
		time = Date.now();
		t = (0.001 * (time - oldTime)) % tau;
		angle = maxRotation * Math.cos(omega * t);

		drawImage(pendulum, pendulumImage, 0, 0, angle);
		drawImage(bob, bobImage, 0, 0, angle);

		if (angle * lastAngle < 0) {
			playSound(clickBuffer);
			count += 1;
			if (diagControl.checked) {
				diags.innerHTML = count;
			}
		}
		lastAngle = angle;
		requestAnimationFrame(draw);
	}
	if (diagControl.checked) {
		diags.innerHTML = count;
	}
	unlockSound();
//	playSound(clickBuffer);
	intervalID = requestAnimationFrame(draw);
};

var insideRates = function (x, y) {
	"use strict";
	return (x > 235) && (x < 307) && (y > 235) && (y < 607);	// needs adjustment
};

bob.onmousemove = null;
bob.ontouchmove = null;

////////////////////////////////////// mouseHasMoved /////////////////////////////////////

var mouseHasMoved = function (x, y) {
	"use strict";
	var value;

//	diags.innerHTML = "hasMoved";

	if (moveEnabled && insideRates(x, y)) {
		value = 781 - Math.round(y / scale);					// needs adjustment

		if (value < 200) {
			value = 200;
		}
		if (value > 500) {
			value = 500;
		}
		bob.vRegP = Math.round(value * scale);

		metroBeatRate = bpm(value);
		beatsPerMin.innerHTML = String(metroBeatRate);
		bpmInput.value = String(metroBeatRate);
//		beatsPerMin.innerHTML = x + ", " + y;

		drawImage(pendulum, pendulumImage, 0, 0, 0);
		moveObj(bob, 251 - 456, 540 - value);
		drawImage(bob, bobImage, 0, 0, 0);
	} else {
		moveEnabled = false;
		setPreference("beatRatePref", String(metroBeatRate));
	}
};

var mouseMove = function (evt) {
	"use strict";
	var x = evt.clientX;		// evt.screenX - screen_X();
	var y = evt.clientY + 89;	// evt.screenY - screen_Y();

	mouseHasMoved(x, y);
};

var touchMove = function (evt) {
	"use strict";
	var x;
	var y;

	evt.preventDefault();

	x = evt.targetTouches[0].clientX;
	y = evt.targetTouches[0].clientY + 89;

	if (evt.targetTouches[0].altKey || altKey.checked) {
		return;
	}

	mouseHasMoved(x, y);
};

/////////////////////////////////////// mouseIsDown //////////////////////////////////////

var mouseIsDown = function (x, y) {
	"use strict";

//	diags.innerHTML = "isDown";

	if (insideRates(x, y)) {
		bob.onmousemove = mouseMove;
		bob.ontouchmove = touchMove;
		moveEnabled = true;
		beatsPerMin.style.color = "#FFFF00";
	} else {
		runEnabled = !runEnabled;
	}
};

bob.onmousedown = function (evt) {
	"use strict";
	var x = evt.clientX;		// evt.screenX - screen_X();
	var y = evt.clientY + 89;	// evt.screenY - screen_Y();

	if (insideRates(x, y) && (evt.altKey || altKey.checked)) {
		playSound(a440Buffer);
		return;
	}
	mouseIsDown(x, y);
};

bob.ontouchstart = function (evt) {
	"use strict";
	var x;
	var y;

	evt.preventDefault();

//	diags.innerHTML = "touchstart";

	x = evt.targetTouches[0].clientX;
	y = evt.targetTouches[0].clientY + 89;

	if (insideRates(x, y) && (evt.targetTouches[0].altKey || altKey.checked)) {
		playSound(a440Buffer);
		return;
	}
	mouseIsDown(x, y);
};

buttons.onmousedown = function (evt) {
	"use strict";
	var x = evt.clientX;		// evt.screenX - screen_X();
	var y = evt.clientY + 89;	// evt.screenY - screen_Y();

	if (evt.altKey || altKey.checked) {
		return;
	}
	mouseIsDown(x, y);
};

/*
buttons.ontouchstart = function (evt) {
	"use strict";
	var x;
	var y;

	evt.preventDefault();

//	diags.innerHTML = "touchstart";

	x = evt.targetTouches[0].clientX;
	y = evt.targetTouches[0].clientY + 89;

	if (evt.targetTouches[0].altKey || altKey.checked) {
		return;
	}
	mouseIsDown(x, y);
};
*/

//////////////////////////////////////// mouseIsUp ///////////////////////////////////////

var mouseIsUp = function () {
	"use strict";

//	diags.innerHTML = "isUp";

	if (moveEnabled) {
		bob.onmousemove = null;
		bob.ontouchmove = null;
		moveEnabled = false;
		setPreference("beatRatePref", String(metroBeatRate));
	} else if (runEnabled) {
		drawImage(pendulum, pendulumImage, 0, 0, maxRotation);
		drawImage(bob, bobImage, 0, 0, maxRotation);
		run();
	} else {
		if (intervalID !== null) {
			intervalID = null;
			//clearInterval(intervalID);
		}
		drawImage(pendulum, pendulumImage, 0, 0, 0);
		drawImage(bob, bobImage, 0, 0, 0);
	}
};

bob.onmouseup = function (evt) {
	"use strict";

	if (evt.altKey || altKey.checked) {
		return;
	}
	mouseIsUp();
};

bob.ontouchend = function (evt) {
	"use strict";
	var x;
	var y;

//	evt.preventDefault();

	x = evt.targetTouches[0].clientX;
	y = evt.targetTouches[0].clientY + 89;

	if (insideRates(x, y) && (evt.targetTouches[0].altKey || altKey.checked)) {
		return;
	}
	mouseIsUp();
};

buttons.onmouseup = function (evt) {
	"use strict";
	if (evt.altKey || altKey.checked) {
		return;
	}
	mouseIsUp();
};

/*
buttons.ontouchend = function (evt) {
	"use strict";

//	evt.preventDefault();

	if (evt.targetTouches[0].altKey || altKey.checked) {
		return;
	}
	mouseIsUp();
};
*/

//////////////////////////////////////////////////////////////////////////////////////////

bpmInput.onchange = function () {
	"use strict";
	var beats = parseInt(bpmInput.value, 10);
	if (!Number.isNaN(beats)) {
		if ((40 <= beats) && (beats <= 208)) {
			metroBeatRate = beats;
			bob.vRegP = Math.round(val(beats) * scale);
			beatsPerMin.innerHTML = String(metroBeatRate);
			beatsPerMin.style.color = "#FFFF00";
			moveObj(bob, 251 - 456, 540 - bob.vRegP);
			drawImage(bob, bobImage, 0, 0, 0);
			setPreference("beatRatePref", String(metroBeatRate));
			bpmInput.value = String(beats);
		}
	}
};

//////////////////////////////////////////////////////////////////////////////////////////

var sampleLength = Number(sampleLengthPref);
sampleLengthMenu.value = sampleLengthPref;
var arrayLength = sampleLength + 1;

var fifo = [];
var on = 0;								// when on == off, the fifo is empty
var off = 0;
var count = 0;							// number of samples accumulated
var oldTime = Date.now();
var oldInterval = 0;
var sum = 0;

sampleLengthMenu.onchange = function () {
	"use strict";
	sampleLengthPref = sampleLengthMenu.value;
	sampleLength = Number(sampleLengthPref);
	arrayLength = sampleLength + 1;
	setPreference("sampleLengthPref", sampleLengthPref);
};

function push(val) {
	"use strict";
	var newOn = (on + 1) % arrayLength;

	if (newOn !== off) {
		fifo[on] = val;
		on = newOn;
	}
}

function pull() {
	"use strict";
	var result;

	if (off !== on) {
		result = fifo[off];
		off = (off + 1) % arrayLength;
		return result;
	}
	return -1;
}

function process(newInterval) {
	"use strict";
	var average;

	if (newInterval > 3000) {
		on = 0;
		off = 0;
		count = 0;
		oldTime = Date.now();
		oldInterval = 0;
		sum = 0;
		newInterval = 652; // 92 bpm
		beatsPerMin.style.color = "#FFFFFF";
	}

	if (count >= sampleLength) {
		oldInterval = pull();
		beatsPerMin.style.color = "#00FF00";
	} else {
		count += 1;
	}

	push(newInterval);
	sum = sum + newInterval - oldInterval;
	average = sum / count;
	beatsPerMin.innerHTML = Math.round(60000 / average);
}

cover.onmousedown = function (evt) {
	"use strict";
	var t;
	var beats;

	if (altKey.checked || evt.altKey) {
		beats = Number(beatsPerMin.innerHTML);
		if (!Number.isNaN(beats)) {
			if ((40 <= beats) && (beats <= 208)) {
				metroBeatRate = beats;
				bob.vRegP = Math.round(val(beats) * scale);
				beatsPerMin.style.color = "#FFFF00";
				bpmInput.value = String(metroBeatRate);
				moveObj(bob, 251 - 456, 540 - bob.vRegP);
				drawImage(bob, bobImage, 0, 0, 0);
				setPreference("beatRatePref", String(metroBeatRate));
			}
		}
		return;
	}

	t = Date.now();
	process(t - oldTime);
	oldTime = t;
};

function start() {
	"use strict";
	drawImage(pendulum, pendulumImage, 0, 0, 0);
	drawImage(bob, bobImage, 0, 0, 0);
}

window.onload = function () {
	initSound();
	setTimeout(start, 1000);
};
