/*
	HexDeCoder - A hexadecimal/decimal/octal/binary converter.
	Copyright © 2004-2012 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
	
	HexDeCoder - version 2.0.1
	2 December, 2012
	Copyright © 2004-2012 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*global convertUTF8toUTF16, add, bitMarkers, decimalFrame, div, fullFrame, ipClass, ipv4, mod,
		 parseInteger, rightJustify, signedDecimal, utf16, utf8, longDiv, longMod
*/

/*properties
    ceil, floor, fromCharCode, indexOf, length, log, match, round, substring,
    toString, toUpperCase, value
*/

var bits = 64; // Do not set to more than 64

var maxBase = 36;

var digitVal = [];

function zero() {
    var r = [], i;
    
    for (i = 0; i < bits; i += 1) {
        r[i] = 0;
    }
    return r;
}

function one() {
    var r = [], i;
    
    for (i = 1; i < bits; i += 1) {
        r[i] = 0;
    }
    r[0] = 1;
    return r;
}

function isZero(r) {
    var i;
    
    for (i = 0; i < bits; i += 1) {
        if (r[i] === 1) {
            return false;
        }
    }
    return true;
}

function makeDigitVal() {
    var i;
    
    digitVal[0] = zero();
    for (i = 1; i < maxBase; i += 1) {
        digitVal[i] = add(digitVal[i - 1], one());
    }
}

function minInteger() { // largest negative integer in two's complement
    var r = [], i;
    
    for (i = 0; i < bits - 1; i += 1) {
        r[i] = 0;
    }
    r[bits - 1] = 1;
    return r;
}

function isMinInteger(a) { // largest negative integer in two's complement
    var i;
    
    for (i = 0; i < bits - 1; i += 1) {
        if (a[i] !== 0) {
            return false;
        }
    }
    if (a[bits - 1] !== 1) {
        return false;
    }
    return true;
}

function neg(a) { // two's complement of a
    var r, found, i;
    
    if (isNaN(a[0])) {
        return [NaN];
    }
    if (isMinInteger(a)) {
        return [NaN];
    } // can't get neg of minInteger
    r = [];
    found = false;
    for (i = 0; i < bits; i += 1) {
        if (found) {
            r[i] = 1 - a[i];
        } else {
            r[i] = a[i];
            found = (a[i] === 1);
        }
    }
    return r;
}

function frameSize(bits, base) {
    return Math.ceil(bits * Math.log(2) / Math.log(base)); // 2^bits === base^frameSize
}

function classOfAddress(a) {
    if (a[31] === 0) {
        return 'Class A';
    }
    if (a[30] === 0) {
        return 'Class B';
    }
    if (a[29] === 0) {
        return 'Class C';
    }
    if (a[28] === 0) {
        return 'Class D';
    }
    return 'Class E';
}

function toIPv4Network(a) {
    var r = [], n, v, i;
    
    for (n = 0; n < 4; n += 1) {
        v = 0;
        for (i = 8 * n; i < 8 * n + 8; i += 1) {
            v = 2 * v + a[31 - i];
        }
        r[n] = String(v);
    }
    if (a[31] === 0) {				// Class A
        r[1] = r[2] = r[3] = '0';
    } else if (a[30] === 0) { 		// Class B
        r[2] = r[3] = '0';
    } else if (a[29] === 0) { 		// Class C
        r[3] = '0';
    }
    return r[0] + '.' + r[1] + '.' + r[2] + '.' + r[3];
}

function toIPv4String(a) {
    var r = "", n, v, i;
    
    for (n = 0; n < 4; n += 1) {
        v = 0;
        for (i = 8 * n; i < 8 * n + 8; i += 1) {
            v = 2 * v + a[31 - i];
        }
        if (n !== 0) {
            r += '.';
        }
        r += String(v);
    }
    return r;
}

function justify(s, base) {
    if (rightJustify || ((base === 2) && bitMarkers)) {
        while (s.length < bits) {
            s = " " + s;
        }
    }
    return s;
}

function unJustify(s) {
    while (s[0] === " ") {
        s = s.substring(1);
    }
    return s;
}

function toDigitString(a, base) {
    var digit, c, r, s, frameMax;

    if (isNaN(a[0])) {
        return "";
    }

    if (base === ipv4) {
        ipClass.value = classOfAddress(a) + ': ' + toIPv4Network(a);
        return toIPv4String(a);
    }

    if (base === utf16) {
        return justify(String.fromCharCode(Number(toDigitString(a, 10)) % 0xFFFF), base);
    }
    if (base === utf8) {
        return justify(String.fromCharCode(convertUTF8toUTF16(Number(toDigitString(a, 10)) % 0xFFFFFF)), base);
    }

    if ((base === 10) && signedDecimal && (a[bits - 1] === 1)) { // number is negative
        if (isMinInteger(a)) {
            if (bits === 64) {
                return justify("-9223372036854775808", base);
            }
            if (bits === 32) {
                return justify("-2147483648", base);
            }
            if (bits === 16) {
                return justify("-32768", base);
            }
            return "Error";
        }
        return justify("-" + unJustify(toDigitString(neg(a), 10)), base);
    }

    digit = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    c = mod(a, base);
    r = div(a, base);
    s = digit[c];
    frameMax = 0;

    while (!isZero(r)) {
        c = mod(r, base);
        r = div(r, base);
        s = digit[c] + s;
    }

    frameMax = frameSize(bits, base);

    if ((fullFrame && (base !== 10)) || (decimalFrame && (base === 10))) {
        while (s.length < frameMax) {
            s = "0" + s;
        }
    }

    return justify(s, base);
}

function toBitString(a) {
    if (isNaN(a[0])) {
        return "NaN";
    }
    return toDigitString(a, 2);
}

function parseIPv4(data) {
    var found, val, i, num;
    
    if (data.length === "") {
        return [NaN];
    }

    found = data.match(/(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3}).*$/);
    if (found === null) {
        return [NaN];
    }

    val = 0;
    for (i = 1; i < 5; i += 1) {
        num = Number(found[i]);
        if (num > 255) {
            return [NaN];
        }
        val = 256 * val + num;
    }
    return parseInteger(val.toString(16), 16);
}

function val(s) { // converts string of bits to array of bits
    return parseInteger(s, 2);
}

function mul(a, b) { // multiply a by b, b in range 0..63 only
	var r = [], i;

    if (isNaN(a[0]) || isNaN(b)) {
        return [NaN];
    }
    if (b > 63) {
        return [NaN];
    }

    switch (b) {
    case 2:
        if (a[bits - 1] !== 0) {
            return [NaN];
        }
        for (i = bits - 1; i > 0; i = i - 1) {
            r[i] = a[i - 1];
        }
        r[0] = 0;
        return r;
    //break;

    case 4:
        if ((a[bits - 1] !== 0) || (a[bits - 2] !== 0)) {
            return [NaN];
        }
        for (i = bits - 1; i > 1; i = i - 1) {
            r[i] = a[i - 2];
        }
        r[0] = r[1] = 0;
        return r;
    //break;

    case 8:
        if ((a[bits - 1] !== 0) || (a[bits - 2] !== 0) || (a[bits - 3] !== 0)) {
            return [NaN];
        }
        for (i = bits - 1; i > 2; i = i - 1) {
            r[i] = a[i - 3];
        }
        r[0] = r[1] = r[2] = 0;
        return r;
    //break;

    case 16:
        if ((a[bits - 1] !== 0) || (a[bits - 2] !== 0) || (a[bits - 3] !== 0) || (a[bits - 4] !== 0)) {
            return [NaN];
        }
        for (i = bits - 1; i > 3; i = i - 1) {
            r[i] = a[i - 4];
        }
        r[0] = r[1] = r[2] = r[3] = 0;
        return r;
    //break;

    case 32:
        if ((a[bits - 1] !== 0) || (a[bits - 2] !== 0) || (a[bits - 3] !== 0) || (a[bits - 4] !== 0) || (a[bits - 5] !== 0)) {
            return [NaN];
        }
        for (i = bits - 1; i > 4; i = i - 1) {
            r[i] = a[i - 5];
        }
        r[0] = r[1] = r[2] = r[3] = r[4] = 0;
        return r;
    //break;

    default:
        r = zero();
        if ((b & 1) !== 0) {
            r = add(r, a);
        }
        if ((b & 2) !== 0) {
            r = add(r, mul(a, 2));
        }
        if ((b & 4) !== 0) {
            r = add(r, mul(a, 4));
        }
        if ((b & 8) !== 0) {
            r = add(r, mul(a, 8));
        }
        if ((b & 16) !== 0) {
            r = add(r, mul(a, 16));
        }
        if ((b & 32) !== 0) {
            r = add(r, mul(a, 32));
        }
        return r;
        //break;
    }
}

function parseInteger(s, base) { // converts string of digits to array of bits
	var n, frameMax, valid, c, r, i;

    if (s[0] === "-") {
        if ((base !== 10) || (!signedDecimal)) {
            return [NaN];
        }
        if (s.substring(0, 2) === "--") {
            return [NaN];
        }
    }

    if ((base === 10) && signedDecimal && (s[0] === "-")) {
        if ((bits === 64) && (s === "-9223372036854775808")) {
            return minInteger();
        }
        if ((bits === 32) && (s === "-2147483648")) {
            return minInteger();
        }
        if ((bits === 16) && (s === "-32768")) {
            return minInteger();
        }

        return neg(parseInteger(s.substring(1), 10));
    }

    n = s.length;
    if (n === 0) {
        return [NaN];
    }
    frameMax = frameSize(bits, base);
    if (n > frameMax) {
        return [NaN];
    }
    s = s.toUpperCase();
    valid = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ'.substring(0, base);
    r = zero();
    for (i = 0; i < n; i += 1) {
        if ((c = valid.indexOf(s[i])) < 0) {
            return [NaN];
        }
        r = add(mul(r, base), digitVal[c]);
    }
    if (signedDecimal && (base === 10) && (r[bits - 1] === 1)) {
        return [NaN];
    }
    return r;
}

function add(a, b) {
    var r = [], carry = 0, sum, i;

    if (isNaN(a[0]) || isNaN(b[0])) {
        return [NaN];
    }
    for (i = 0; i < bits; i += 1) {
        sum = a[i] + b[i] + carry;
        r[i] = sum % 2;
        if (sum >= 2) {
            carry = 1;
        } else {
            carry = 0;
        }
    }
    if (carry === 1) {
        return [NaN];
    }
    return r;
}

function div(a, b) { // divide a by b, b in range 0..63 only
    var r = [], i;

    if (isNaN(a[0]) || isNaN(b)) {
        return [NaN];
    }
    if (b > 63) {
        return [NaN];
    }

    switch (b) {
    case 2:
        for (i = 0; i < bits - 1; i += 1) {
            r[i] = a[i + 1];
        }
        r[bits - 1] = 0;
        return r;
    //break;

    case 4:
        for (i = 0; i < bits - 2; i += 1) {
            r[i] = a[i + 2];
        }
        r[bits - 1] = r[bits - 2] = 0;
        return r;
    //break;

    case 8:
        for (i = 0; i < bits - 3; i += 1) {
            r[i] = a[i + 3];
        }
        r[bits - 1] = r[bits - 2] = r[bits - 3] = 0;
        return r;
    //break;

    case 16:
        for (i = 0; i < bits - 4; i += 1) {
            r[i] = a[i + 4];
        }
        r[bits - 1] = r[bits - 2] = r[bits - 3] = r[bits - 4] = 0;
        return r;
    //break;

    case 32:
        for (i = 0; i < bits - 5; i += 1) {
            r[i] = a[i + 5];
        }
        r[bits - 1] = r[bits - 2] = r[bits - 3] = r[bits - 4] = r[bits - 5] = 0;
        return r;
    //break;

    default:
        return longDiv(a, b);
        //break;
    }
}

function mod(a, b) { // modulo of a divided by b, b in range 0..63 only
	var r;
	
    if (isNaN(a[0]) || isNaN(b)) {
        return NaN;
    }
    if (b > 63) {
        return [NaN];
    }

    switch (b) {
    case 2:
        return a[0];
    //break;

    case 4:
        return 2 * a[1] + a[0];
    //break;

    case 8:
        return 4 * a[2] + 2 * a[1] + a[0];
    //break;

    case 16:
        return 8 * a[3] + 4 * a[2] + 2 * a[1] + a[0];
    //break;

    case 32:
        return 16 * a[4] + 8 * a[3] + 4 * a[2] + 2 * a[1] + a[0];
    //break;

    default:
        r = longMod(a, b);
        return 32 * r[5] + 16 * r[4] + 8 * r[3] + 4 * r[2] + 2 * r[1] + r[0];
        //break;
    }
}

function split(a) { // split a into an array of 32 bit numbers
    var r = [], i;

    if (isNaN(a[0])) {
        return [NaN];
    }

    if (bits <= 32) {
        r[0] = 0;
        for (i = bits - 1; i >= 0; i = i - 1) {
            r[0] = 2 * r[0] + a[i];
        }
        r[1] = 0;
    } else if (bits <= 64) {
        r[0] = 0;
        for (i = 31; i >= 0; i = i - 1) {
            r[0] = 2 * r[0] + a[i];
        }

        r[1] = 0;
        for (i = bits - 1; i >= 32; i = i - 1) {
            r[1] = 2 * r[1] + a[i];
        }
    }
    return r;
}

function join(a) {	// join array of 32 bit numbers into array of bits
	var r = [], x, i;
	
	if (isNaN(a[0])) {
        return [NaN];
    }

    x = a[0];
    if (bits <= 32) {
        for (i = 0; i < bits; i += 1) {
            r[i] = x % 2;
            x = x >>> 1;
        }
    } else if (bits <= 64) {
        for (i = 0; i < 32; i += 1) {
            r[i] = x % 2;
            x = x >>> 1;
        }
        x = a[1];
        for (i = 32; i < bits; i += 1) {
            r[i] = x % 2;
            x = x >>> 1;
        }
    }
    return r;
}

function longDiv(a, b) {
	var x, lo, hi, hiMod, hiDiv, loDiv, r;

    if (isNaN(a[0]) || isNaN(b)) {
        return [NaN];
    }

    x = split(a);
    lo = x[0];
    hi = x[1];

    hiMod = hi % b;
    hiDiv = Math.round((hi - hiMod) / b);

    lo += 4294967296 * hiMod;

    loDiv = Math.floor(lo / b);
    //loMod = lo - b * loDiv;

    r = [];
    r[0] = loDiv;
    r[1] = hiDiv;

    return join(r);
}

function longMod(a, b) {
    var x, lo, hi, hiMod, loDiv, loMod, r;
    
    if (isNaN(a[0]) || isNaN(b)) {
        return [NaN];
    }

    x = split(a);
    lo = x[0];
    hi = x[1];

    hiMod = hi % b;
    //hiDiv = Math.round((hi - hiMod) / b);

    lo += 4294967296 * hiMod;

    loDiv = Math.floor(lo / b);
    loMod = lo - b * loDiv;

    r = [];
    r[0] = loMod;
    r[1] = 0;

    return join(r);
}