/*
	HexDeCoder - A hexadecimal/decimal/octal/binary converter.
	Copyright © 2004-2015 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

	HexDeCoder - browser version 1.0
	28 April, 2015
	Copyright © 2004-2015 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*global paramOne, paramTwo, paramThree, paramFour,
	headingOne, headingTwo, headingThree, headingFour,
	clearData, signedDecimal, Menu, curWidth:true, minWidth, maxWidth,
	setAmber, setTooltips, setUp, helpButton, dragEnabled:true,
	fullFrameBox, deciFrameBox, rightJustifyBox, bitMarkersBox,
	bitNumbers, byteMarks,
	
	parseInteger, unJustify, toDigitString, frameSize, bits:true, ipClass,  parseIPv4,
	classOfAddress, toIPv4Network, makeDigitVal
*/

/*properties
    charCodeAt, checked, indexOf, length, onchange, onkeyup, opacity, style,
    substring, title, toString, toUpperCase, value
*/

// map custom names onto model names
var hexadecimal = paramOne;
var decimal     = paramTwo;
var custom      = paramThree;
var binary      = paramFour;

var customBaseHeading  = headingThree;

var wordLengthMenu   = Menu[0];
var customBaseMenu   = Menu[1];
var bitNumberingMenu = Menu[2];

var dataArray = [];

var customBase = 8;

var utf16 = 42; // base 42 is used as a flag for UTF-16 data
var utf8 = 43; // base 43 is used as a flag for UTF-8	data
var ipv4 = 44; // base 44 is used as a flag for IPv4	data

// these are the substitute preference values
var wordLengthValue = '64-bit';
var customBaseValue = '8';
//var signedDecimal = false;	// defines in medel.js
var fullFrame    = false;
var decimalFrame = false;
var rightJustify = false;
var bitMarkers   = false;
var bigEndian    = true;

var bitNumberDataBE = "0123456789012345678901234567890123456789012345678901234567890123";
var byteMarksDataBE = "|0     7|8    15|16   23|24   31|32   39|40   47|48   55|56   63|";

var bitNumberDataLE = "3210987654321098765432109876543210987654321098765432109876543210";
var byteMarksDataLE = "|63   56|55   48|47   40|39   32|31   24|23   16|15    8|7     0|";

function recomputeData() {
    if (hexadecimal.value !== "") {
        dataArray = parseInteger(unJustify(hexadecimal.value), 16);

        if (isNaN(dataArray[0])) {
            clearData();
        } else {
            //hexadecimal.value  = toDigitString(dataArray, 16);
            decimal.value = toDigitString(dataArray, 10);
            custom.value = toDigitString(dataArray, customBase);
            binary.value = toDigitString(dataArray, 2);
        }
    }
}

function updatePrefs() {
    if (wordLengthValue === "16-bit") {
        bits = 16;
    } else if (wordLengthValue === "32-bit") {
        bits = 32;
    } else if (wordLengthValue === "64-bit") {
        bits = 64;
    }

    if (customBaseValue === "UTF-16") {
        customBase = utf16;
    } else if (customBaseValue === "UTF-8") {
        customBase = utf8;
    } else if (customBaseValue === "IP-v4") {
        customBase = ipv4;
    } else {
        customBase = Number(customBaseValue);
    }

    if (((customBase === utf8) && (bits === 16)) || (customBase === ipv4)) {
        wordLengthValue = "32-bit";
        bits = 32;
        wordLengthMenu.value = wordLengthValue;
    }

    if (customBase === ipv4) {
        ipClass.style.opacity = 1.0;
        ipClass.value = "";
    } else {
        ipClass.style.opacity = 0;
//		ipClass.bgOpacity = 0;
    }

    if ((customBase === utf16) || (customBase === utf8)) {
        custom.title = "Enter a Unicode character in this box.";
    } else if (customBase === ipv4) {
        custom.title = "Enter an IP address in this box.";
    } else if (customBase === 16) {
        custom.title = "Enter a hexadecimal number in this box.";
    } else if (customBase === 10) {
        custom.title = "Enter a decimal number in this box.";
    } else if (customBase === 8) {
        custom.title = "Enter an octal number in this box.";
    } else if (customBase === 2) {
        custom.title = "Enter a binary number in this box.";
    } else {
        custom.title = "Enter a base " + customBase + " number in this box.";
    }

    if (customBase === utf16) {
        customBaseHeading.value = "UC";
        //customBaseHeading.title = 'Using UTF-16 Encoding.\nClick for UTF-8	Encoding.';
    } else if (customBase === utf8) {
        customBaseHeading.value = "U8";
        //customBaseHeading.title = 'Using UTF-8	 Encoding.\nClick for UTF-16 Encoding.';
    } else if (customBase === ipv4) {
        customBaseHeading.value = "IP";
        //customBaseHeading.title = 'Using IP-v4	 Encoding.';
    } else {
        customBaseHeading.value = customBase;
        //customBaseHeading.title = 'Custom Base is ' + customBase;
    }

    fullFrame    = fullFrameBox.checked;
    decimalFrame = deciFrameBox.checked;
	rightJustify = rightJustifyBox.checked;
    bitMarkers   = bitMarkersBox.checked;
    
//    bigEndian = (preferences.bigEndian.value === "big-endian");

    if (bigEndian) {
        bitNumbers.value = bitNumberDataBE.substring(0, bits);
        byteMarks.value  = byteMarksDataBE.substring(0, bits + 1);
    } else {
        bitNumbers.value = bitNumberDataLE.substring(64 - bits);
        byteMarks.value  = byteMarksDataLE.substring(64 - bits);
    }

    if (bits === 16) {
        curWidth = minWidth;
    } else if (bits === 32) {
        curWidth = minWidth;
    } else if (bits === 64) {
        curWidth = maxWidth;
    }
    helpButton.style.opacity = 1.0;
    dragEnabled = false;

    if (bitMarkers) {
        bitNumbers.style.opacity = byteMarks.style.opacity = 1.0;
    } else {
        bitNumbers.style.opacity = byteMarks.style.opacity = 0.0;
    }

    setAmber(signedDecimal);
    setTooltips(signedDecimal);
    setUp(curWidth);
	recomputeData();
}

function clean(s, base, signed) {
	var rest, frameMax, n, u, valid, r, i;
	
    if (signed && (s[0] === "-")) {
        rest = clean(s.substring(1), base, false);
        if (rest !== "") {
            return "-" + rest;
        }
        return "";
    }

    frameMax = frameSize(bits, base);
    s = s.substring(0, frameMax + 1);
    n = s.length;
    if (n === 0) {
        return "";
    }
    u = s.toUpperCase();
    valid = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ'.substring(0, base);
    r = "";
    for (i = 0; i < n; i += 1) {
        if (valid.indexOf(u[i]) >= 0) {
            r += s[i];
        }
    }
    if (r.length < n) {
        return r;
    }
    return r.substring(0, r.length - 1);
}

function convertUTF16toUTF8(c) {
    //	[0x00000000 - 0x0000007f] [00000000.0bbbbbbb] -> 0bbbbbbb
    //	[0x00000080 - 0x000007ff] [00000bbb.bbbbbbbb] -> 110bbbbb, 10bbbbbb
    //	[0x00000800 - 0x0000ffff] [bbbbbbbb.bbbbbbbb] -> 1110bbbb, 10bbbbbb, 10bbbbbb

    if ((0xD800 <= c) && (c <= 0xDFFF)) {
        return NaN;
    } // surrogate value
    if (c < 0x80) {
        return c;
    }
    if (c < 0x800) {
        return ((0xC0 | ((c >> 6))) << 8) | (0x80 | (c & 0x3F));
    }
    if (c < 0x10000) {
        return ((0xE0 | (c >> 12)) << 16) | ((0x80 | ((c >> 6) & 0x3F)) << 8) | (0x80 | (c & 0x3F));
    }
    return NaN; // illegal UTF16 value
}

function convertUTF8toUTF16(c) {
    var b, r;

    if ((c & 0xFF000000) !== 0) {
        return NaN;
    } // more than 3 bytes

    if ((c & 0x00FF0000) !== 0) { // 3 bytes
        b = (c >> 16);
        if ((b & 0xF0) !== 0xE0) {
            return NaN;
        }
        r = (b & 0x0F) << 12;
        b = (c >> 8) & 0xFF;
        if ((b & 0xC0) !== 0x80) {
            return NaN;
        }
        r |= (b & 0x3F) << 6;
        b = c & 0xFF;
        if ((b & 0xC0) !== 0x80) {
            return NaN;
        }
        r |= (b & 0x3F);
        return r;
    }
    if ((c & 0x0000FF00) !== 0) { // 2 bytes
        b = (c >> 8);
        if ((b & 0xE0) !== 0xC0) {
            return NaN;
        }
        r = (b & 0x1F) << 6;
        b = c & 0xFF;
        if ((b & 0xC0) !== 0x80) {
            return NaN;
        }
        r |= (b & 0x3F);
        return r;
    }
    if ((c & 0x80) !== 0) {
        return NaN;
    }
    return c;
}

function parseUTF16(data) {
    if (data.length !== 1) {
        return [NaN];
    }
    if (customBase === utf16) {
        return parseInteger(data.charCodeAt(0).toString(16), 16);
    }
    if (customBase === utf8) {
        return parseInteger(convertUTF16toUTF8(data.charCodeAt(0)).toString(16), 16);
    }
    return [NaN];
}

function toggleUTF() {
    if (customBase === utf16) {
        customBaseValue = "UTF-8";
        updatePrefs();
    } else if (customBase === utf8) {
        customBaseValue = "UTF-16";
        updatePrefs();
    } else {
        alert("Invalid customBase in toggleUTF(): " + customBase);
    }
}

wordLengthMenu.onchange  = function () {
	wordLengthValue = this.value;
	updatePrefs();
};

customBaseMenu.onchange  = function () {
	customBaseValue = this.value;
	updatePrefs();
};

bitNumberingMenu.onchange  = function () {
	bigEndian = this.value === "big-endian";
	updatePrefs();
};

fullFrameBox.onchange  = function () {
	updatePrefs();
};

deciFrameBox.onchange  = function () {
	updatePrefs();
};

rightJustifyBox.onchange  = function () {
	updatePrefs();
};

bitMarkersBox.onchange  = function () {
	updatePrefs();
};

hexadecimal.onkeyup = function () {
    //testLicense();
    ipClass.value = "";
    hexadecimal.value = unJustify(hexadecimal.value);
    dataArray = parseInteger(hexadecimal.value, 16);
    if (isNaN(dataArray[0])) {
        hexadecimal.value = clean(hexadecimal.value, 16, false);
        if (hexadecimal.value !== "") {
            dataArray = parseInteger(hexadecimal.value, 16);
        }
    }
    //hexadecimal.value = toDigitString(dataArray, 16);
    decimal.value = toDigitString(dataArray, 10);
    custom.value = toDigitString(dataArray, customBase);
    binary.value = toDigitString(dataArray, 2);
};

decimal.onkeyup = function () {
    //testLicense();
    ipClass.value = "";
    decimal.value = unJustify(decimal.value);
    if ((decimal.value !== "-") || (!signedDecimal)) {
        dataArray = parseInteger(decimal.value, 10);
        if (isNaN(dataArray[0])) {
            decimal.value = clean(decimal.value, 10, signedDecimal);
            if (decimal.value !== "") {
                dataArray = parseInteger(decimal.value, 10);
            }
        }
        hexadecimal.value = toDigitString(dataArray, 16);
        //decimal.value = toDigitString(dataArray, 10);
        custom.value = toDigitString(dataArray, customBase);
        binary.value = toDigitString(dataArray, 2);
    } else {
        clearData();
        decimal.value = "-";
    }
};

custom.onkeyup = function () {
    //testLicense();
    if (customBase === ipv4) {
        custom.value = unJustify(custom.value);
        dataArray = parseIPv4(custom.value);
        hexadecimal.value = toDigitString(dataArray, 16);
        decimal.value = toDigitString(dataArray, 10);
        if (!isNaN(dataArray[0])) {
            custom.value = toDigitString(dataArray, ipv4);
            ipClass.value = classOfAddress(dataArray) + ': ' + toIPv4Network(dataArray);
        } else {
            ipClass.value = "";
        }
        binary.value = toDigitString(dataArray, 2);
    } else if ((customBase === utf16) || (customBase === utf8)) {
        if (custom.value !== " ") {
            custom.value = unJustify(custom.value);
        }
        dataArray = parseUTF16(custom.value);
        if (isNaN(dataArray[0])) {
            if (custom.value !== "") {
                custom.value = custom.value[0];
                dataArray = parseUTF16(custom.value);
            }
        }
        hexadecimal.value = toDigitString(dataArray, 16);
        decimal.value = toDigitString(dataArray, 10);
        //custom.value  = toDigitString(dataArray, customBase);
        binary.value = toDigitString(dataArray, 2);
    } else {
        custom.value = unJustify(custom.value);
        if ((customBase !== 10) || (custom.value !== "-") || (!signedDecimal)) {
            dataArray = parseInteger(custom.value, customBase);
            if (isNaN(dataArray[0])) {
                custom.value = clean(custom.value, customBase, ((customBase === 10) && signedDecimal));
                if (custom.value !== "") {
                    dataArray = parseInteger(custom.value, customBase);
                }
            }
            hexadecimal.value = toDigitString(dataArray, 16);
            decimal.value = toDigitString(dataArray, 10);
            //custom.value  = toDigitString(dataArray, customBase);
            binary.value = toDigitString(dataArray, 2);
        } else {
            clearData();
            custom.value = "-";
        }
    }
};

binary.onkeyup = function () {
    //testLicense();
    ipClass.value = "";
    binary.value = unJustify(binary.value);
    dataArray = parseInteger(binary.value, 2);
    if (isNaN(dataArray[0])) {
        binary.value = clean(binary.value, 2);
        if (binary.value !== "") {
            dataArray = parseInteger(binary.value, 2);
        }
    }
    hexadecimal.value = toDigitString(dataArray, 16);
    decimal.value = toDigitString(dataArray, 10);
    custom.value = toDigitString(dataArray, customBase);
    //binary.value  = toDigitString(dataArray,  2);
};

makeDigitVal();
updatePrefs();
