/*
    Hex Display - A Hex Display Widget
    Copyright © 2008-2019 Harry Whitfield

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License as
	published by the Free Software Foundation; either version 2 of
	the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public
	License along with this program; if not, write to the Free
	Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
	MA  02110-1301  USA

	Hex Display - version 1.0 for Hex Display Web widget
	14 October, 2019
	Copyright © 2008-2019 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*jslint browser, devel, for */

/*property
    DONE, addEventListener, appendChild, body, byteLength, click, createElement,
    dataTransfer, display, download, dropEffect, files, floor, getElementById,
    href, innerHTML, isNaN, keyCode, lastModified, lastModifiedDate, length,
    name, onclick, onkeypress, onload, onloadend, opacity, play, preventDefault,
    readAsArrayBuffer, readyState, removeChild, replace, result, size, slice,
    stopPropagation, style, substring, target, toISOString, toString,
    toUpperCase, type, value, which
*/

var window;
var document;
var File;
var FileReader;
var FileList;
var Blob;

var dropArrow = document.getElementById("dropArrow");
var dropZone = document.getElementById("dropZone");
var fileInput = document.getElementById("fileInput");
var nameArea = document.getElementById("nameArea");
var displayArea = document.getElementById("displayArea");
var start_address = document.getElementById("start_address");
var end_address = document.getElementById("end_address");
var display_button = document.getElementById("display_button");
var first_block = document.getElementById("first_block");
var next_block = document.getElementById("next_block");
var prev_block = document.getElementById("prev_block");
var last_block = document.getElementById("last_block");
var download_display = document.getElementById("download_display");
var enable_sound = document.getElementById("enable_sound");

var makeHex;
var bytes = [];
var maxBytes = 0x400;
var maxBytes2 = 0x1000;

var Audio;
var mistake = new Audio("mistake.wav");

function beep() {
	if (enable_sound.checked) {
		mistake.play();
	}
}

function min(a, b) {
	return (
		b < a
		? b
		: a
	);
}

var filesystem = {};

filesystem.download = function (filename, text) {
  	var element = document.createElement("a");

  	element.href = "data:text/plain;charset=utf-8," + encodeURIComponent(text);
  	element.download = filename;
  	element.style.display = "none";
  	document.body.appendChild(element);
  	element.click();
  	document.body.removeChild(element);
};

function isoTime() {
	var temp = new Date().toISOString().replace(/\D/g, "").substring(0, 14);
	var date = temp.substring(0, 8);
	var time = temp.substring(8);

	return date + "T" + time + "Z";
}

function write(s) {
	s = s.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;");

	displayArea.innerHTML = "<pre>" + s.replace(/\n/g, "<br>") + "</pre>";
}

function addressesOK(start, end) {
	if (nameArea.innerHTML === "File name:") {
		beep();
		alert("Please load a file!");
		return false;
	}

	if (Number.isNaN(start) || (start < 0)) {
		beep();
		alert("Invalid start address!");
		return false;
	}

	if (Number.isNaN(end) || (end < 0)) {
		beep();
		alert("Invalid end address!");
		return false;
	}

	if (start >= end) {
		beep();
		alert("Invalid addresses - start must be less than end");
		return false;
	}
	return true;
}


function process() {
	var start = parseInt(start_address.value);
	var end = parseInt(end_address.value);

	if (!addressesOK(start, end)) {
		return;
	}

	if (start < bytes.length) {
		write("");
		write(makeHex(bytes, start, min(maxBytes2, min(end - start, bytes.length - start))));
		return;
	}
	beep();
}

start_address.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	if (x === 13) {
	    process();
	}
};

end_address.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	if (x === 13) {
	    process();
	}
};

display_button.onclick = function () {
	process();
};

first_block.onclick = function () {
	var start = parseInt(start_address.value);
	var end = parseInt(end_address.value);

	if (!addressesOK(start, end)) {
		return;
	}

	if (bytes.length === 0) {
		beep();
		return;
	}

	start_address.value = "0";
	end_address.value = "0x" + min(bytes.length, 0x1000).toString(16).toUpperCase();
	process();
};

next_block.onclick = function () {
	var start = parseInt(start_address.value);
	var end = parseInt(end_address.value);
	var nextStart;

	if (!addressesOK(start, end)) {
		return;
	}

	if (bytes.length === 0) {
		beep();
		return;
	}

	nextStart = start + 0x1000;
	if (nextStart < bytes.length) {
		start_address.value = "0x" + nextStart.toString(16).toUpperCase();
		end_address.value = "0x" + min(bytes.length, nextStart + 0x1000).toString(16).toUpperCase();
		process();
		return;
	}
	beep();
};

prev_block.onclick = function () {
	var start = parseInt(start_address.value);
	var end = parseInt(end_address.value);
	var nextStart;

	if (!addressesOK(start, end)) {
		return;
	}

	if (bytes.length === 0) {
		beep();
		return;
	}

	nextStart = start - 0x1000;
	if (nextStart >= 0) {
		start_address.value = (
			nextStart === 0
			? "0"
			: "0x" + nextStart.toString(16).toUpperCase()
		);
		end_address.value = "0x" + min(bytes.length, nextStart + 0x1000).toString(16).toUpperCase();
		process();
		return;
	}
	beep();
};

last_block.onclick = function () {
	var start = parseInt(start_address.value);
	var end = parseInt(end_address.value);

	if (!addressesOK(start, end)) {
		return;
	}

	if (bytes.length === 0) {
		beep();
		return;
	}

	var blocks = Math.floor(bytes.length / 0x1000);
	var blockAddress = blocks * 0x1000;

	start_address.value = (
		blocks === 0
		? "0"
		: "0x" + blockAddress.toString(16).toUpperCase()
	);
	end_address.value = "0x" + bytes.length.toString(16).toUpperCase();
	process();
};

download_display.onclick = function () {
	var start = parseInt(start_address.value);
	var end = parseInt(end_address.value);
	var contents = nameArea.innerHTML.replace(/<br>/g, "\n");

	if (!addressesOK(start, end)) {
		return;
	}

	if (contents !== "File name:") {
		if (bytes.length > 0) {
			contents += "\n\n" + makeHex(bytes, start, min(maxBytes2, min(end - start, bytes.length - start)));
		}
		filesystem.download("Hex_Display " + isoTime() + ".txt", contents);
		return;
	}
	beep();
};

window.onload = function () {
	var action = function (b) {
		if (b.byteLength !== 0) {
			bytes = new Uint8Array(b);
			start_address.value = "0";
			end_address.value = "0x" + min(bytes.length, maxBytes).toString(16);
			write(makeHex(bytes, 0, Number(end_address.value)));
			return;
		}
		start_address.value = "0";
		end_address.value = "0";
		write("");
		bytes = [];
	};
/*
	var readImage = function (file) {
		var reader = new FileReader();

		reader.onload = function () {
			var img = new Image();

			dropZone.innerHTML = "";
			img.src = reader.result;
			dropZone.appendChild(img);
		};
		reader.readAsDataURL(file);
	};
*/
	var readBlob = function (file, start, bytes, callback) {
		var reader = new FileReader();
		var blob = file.slice(start, start + bytes);

		reader.onloadend = function (event) {
			if (event.target.readyState === FileReader.DONE) {
				callback(event.target.result);
			}
		};
		reader.readAsArrayBuffer(blob);
	};

	var handleDrop = function (evt) {
        var files = evt.dataTransfer.files;
        var file = files[0];

        evt.stopPropagation();
        evt.preventDefault();
        dropArrow.style.opacity = 1;
		nameArea.innerHTML = (
			"File name: " + file.name + "<br>" +
			"Size: " + file.size + " (0x" + file.size.toString(16).toUpperCase() + ") bytes<br>" +
			"MIME type: " + file.type + "<br>" +
			"Modified: " + new Date(file.lastModified || file.lastModifiedDate)
        );
		readBlob(file, 0, file.size, action);
	};

    var handleDragOver = function (evt) {
        evt.stopPropagation();
        evt.preventDefault();
        evt.dataTransfer.dropEffect = "copy";
        dropArrow.style.opacity = 0;
    };

	var handleDragLeave = function (evt) {
        evt.stopPropagation();
        evt.preventDefault();
        dropArrow.style.opacity = 1;
    };

	var handleSelect = function (evt) {
		var file = fileInput.files[0];

        evt.stopPropagation();
        evt.preventDefault();
		nameArea.innerHTML = (
			"File name: " + file.name + "<br>" +
			"Size: " + file.size + " (0x" + file.size.toString(16).toUpperCase() + ") bytes<br>" +
			"MIME type: " + file.type + "<br>" +
			"Modified: " + new Date(file.lastModified || file.lastModifiedDate)
        );
		readBlob(file, 0, file.size, action);
	};

    if (File && FileReader && FileList && Blob) {
        dropZone.addEventListener("dragover", handleDragOver, false);
        dropZone.addEventListener("drop", handleDrop, false);
       	dropZone.addEventListener("dragleave", handleDragLeave, false);
        fileInput.addEventListener("change", handleSelect, false);
    } else {
        alert("File APIs are missing.");
    }
};
