/*
	Image Size - Displays the dimensions of an image.
	Copyright © 2011-2019 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA 02110-1301	USA

	Image Size - browser version 1.2
	22 November, 2019
	Copyright © 2011-2019 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*jslint browser, devel */

/* global Image, File, FileList, Blob */

/*property
    addEventListener, appendChild, checked, childNodes, close, dataTransfer,
    dropEffect, files, hasChildNodes, height, innerHTML, lastModified,
    lastModifiedDate, match, name, naturalHeight, naturalWidth, onchange,
    onclick, onload, onmousedown, onmouseup, opacity, open, preventDefault,
    readAsDataURL, removeChild, result, round, setAttribute, size, src,
    stopPropagation, style, title, type, value, width
*/

import {newImage, newText, newInput, newFileSelector, newDiv} from "./webWidget.js";

//////////////////////////////////////// GUI setup ///////////////////////////////////////

const style = "font-family:Helvetica;color:black;font-weight:normal;background-color:transparent;border:none;font-size:16px;text-align:center;";

let bg = newImage(0, 0, 108, 72, "Resources/Images/bg.png", 1, 1);
bg.title = "";

let redButton = newImage(3, 4, 14, 14, "Resources/Images/red.png", 2, 1);
redButton.title = "Close the widget.";

let naturalSize = newText(0, 28, 108, 24, "no file", 2, style);
naturalSize.title = "Natural Size of Image.";

let thumbKey = newInput(88, 1, 8, 16, "thumbnail", 2);
thumbKey.setAttribute("type", "checkbox");
thumbKey.title = "Choose thumbnail mode.";
thumbKey.checked = true;					// initially

let thumbButton = newInput(18, 1, 8, 16, "thumbnail", 2);
thumbButton.setAttribute("type", "button");
thumbButton.title = "Choose thumbnail mode.";

let helpButton = newImage(130, 0, 30, 22, "Resources/Images/help.png", 1, 1);
helpButton.title = "Displays help information.";

let dropArrow = newImage(350, 84, 60, 60, "Resources/Images/dropArrow.png", 1, 1);
dropArrow.title = "Drop an image file onto the dropzone immediately below me.";

redButton.onmouseup = function () {
	window.close();
};

helpButton.onmousedown = function () {
	helpButton.style.opacity = "0.5";
};
helpButton.onmouseup = function () {
	helpButton.style.opacity = "1.0";
	window.open("Help.html");
};

/////////////////////////////////////// File Handlers ////////////////////////////////////

let handleFiles = function (fileInput, fileInfo, dropZone, callback) {

	let handleFile = function (file) {
		//let extn = xtn(file.name).toLowerCase();
		let fileType = file.type;
		let reader;

		fileInfo.innerHTML = (
			file.name + ", " + file.size + " bytes, Type: " + fileType +
			",<br>Modified: " + new Date(file.lastModified || file.lastModifiedDate)
		);

		if (fileType.match(/^image\//)) {
			reader = new FileReader();

			reader.onload = function () {
				let img = new Image();
				let delay = function () {
					callback(img, true);
				};
				img.src = reader.result;
				setTimeout(delay, 1000);
			};
			reader.readAsDataURL(file);
		} else {
			callback("Invalid file type: " + fileType, false);
		}
		dropArrow.style.opacity = 1;
	};

	let handleDrop = function (evt) {
		let files = evt.dataTransfer.files;
		let file = files[0];

		evt.stopPropagation();
		evt.preventDefault();
		handleFile(file);
	};

	let handleDragOver = function (evt) {
		evt.stopPropagation();
		evt.preventDefault();
		evt.dataTransfer.dropEffect = "copy";
		dropArrow.style.opacity = 0;
	};

	let handleDragLeave = function (evt) {
		evt.stopPropagation();
		evt.preventDefault();
		dropArrow.style.opacity = 1;
	};

	let handleSelect = function (evt) {
		let file = fileInput.files[0];

		evt.stopPropagation();
		evt.preventDefault();
		handleFile(file);
	};

	if (File && FileReader && FileList && Blob) {
		dropZone.addEventListener("dragover", handleDragOver, false);
		dropZone.addEventListener("drop", handleDrop, false);
		dropZone.addEventListener("dragleave", handleDragLeave, false);
		fileInput.addEventListener("change", handleSelect, false);
	} else {
		alert("File APIs are missing.");
	}
};

//////////////////////////////////////////////////////////////////////////////////////////

let fileInput;
let fileInfo;
let imageArea;
let gImage = null;

function scaleImage(img) {
	let maxWidth = 320;
	let maxHeight = 240;
	let width = img.naturalWidth;
	let height = img.naturalHeight;
	let aspectRatio;

	if (thumbKey.checked) {
		aspectRatio = width / height;

		if (width > maxWidth) {
			width = maxWidth;
			height = Math.round(width / aspectRatio);
		}

		if (height > maxHeight) {
			height = maxHeight;
			width = Math.round(height * aspectRatio);
		}
	}
	img.width = width;
	img.height = height;
}

function changeMode() {
	if (gImage) {
		scaleImage(gImage);
	}
	thumbButton.value = (
		thumbKey.checked
		? "thumbnail"
		: "full-sized"
	);
	thumbButton.title = "Choose " + (
		thumbKey.checked
		? "full-sized"
		: "thumbnail"
	) + " mode.";
	thumbKey.title = "Choose " + (
		thumbKey.checked
		? "full-sized"
		: "thumbnail"
	) + " mode.";
}

thumbKey.onchange = function () {
	changeMode();
};

thumbButton.onclick = function () {
	thumbKey.checked = !thumbKey.checked;
	changeMode();
};

let clearImageArea = function () {
	while (imageArea.hasChildNodes()) {
		imageArea.removeChild(imageArea.childNodes[0]);
	}
};

let callback = function (img, success) {
	let width;
	let height;

	clearImageArea();

	if (success) {
		width = img.naturalWidth;
		height = img.naturalHeight;

		if ((width === undefined) || (width < 10) || (height === undefined) || (height < 10)) {
			alert("Loading failed - try again! " + width + ", " + height);
			naturalSize.innerHTML = "no file";
		} else {
			naturalSize.innerHTML = width + ", " + height;
			gImage = img;
			scaleImage(img);
			imageArea.appendChild(img);
		}
	} else {
		naturalSize.innerHTML = "no file";
		gImage = null;
		alert(img);
	}
};

let fileInfoX = 180;
let fileInfoY = 20;
let fileSelectorY = 96;
let imageAreaY = 144;

window.onload = function () {
	fileInput = newFileSelector(0, fileSelectorY, 200, 6.3, "", 1);
	fileInput.title = "Choose an image file or drop an image file onto the dropzone below me.";

	fileInfo = newDiv(fileInfoX, fileInfoY, 1, "fileInfo");
	fileInfo.title = "File Information";

	imageArea = newDiv(0, imageAreaY, 1, "imageArea");
	imageArea.title = "Image Area and Drop Zone";

	handleFiles(fileInput, fileInfo, imageArea, callback);
	fileInfo.innerHTML = "";
};
