/*
	Kaprekar Sequence Calculator
 	Copyright © 2023 Harry Whitfield

 	This program is free software; you can redistribute it and/or modify it
 	under the terms of the GNU General Public License as published by the
 	Free Software Foundation; either version 2 of the License, or (at your
 	option) any later version.

 	This program is distributed in the hope that it will be useful, but
 	WITHOUT ANY WARRANTY; without even the implied warranty of
 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 	General Public License for more details.

 	You should have received a copy of the GNU General Public License along
 	with this program; if not, write to the Free Software Foundation, Inc.,
 	51 Franklin St, Fifth Floor, Boston, MA	 02110-1301	 USA

 	Kaprekar Sequence Calculator - browser version 1.0.4
 	28 February, 2023
 	Copyright © 2023 Harry Whitfield
 	mailto:g6auc@arrl.net
*/

/*jslint browser, this */

/*global paramOne, paramTwo, paramThree, paramFour,
	headingOne, headingTwo, headingThree, headingFour, clearData,
	execButton
*/

/*property
    concat, forEach, indexOf, isNaN, keyCode, length, onkeydown, onkeypress,
    onmousedown, onmouseup, opacity, preventDefault, push, reverse, select,
    sort, stopPropagation, style, value, which
*/

var gWidgetName;	// global

// map custom names onto model names
var numberBase = paramOne;
var digitsLength = paramTwo;
var firstNumber = paramThree;
var calculatedSequence = paramFour;

var numberBaseHeading = headingOne;
var digitsLengthHeading = headingTwo;
var firstNumberHeading = headingThree;
var calculatedSequenceHeading = headingFour;

/*
Number.prototype.inRange = function (a, b) {
    return (this >= a) && (this <= b);
}
*/

/////////////////////////////////////////////////////////////////////////////////////////

function val(arr, base) {				// convert array of digit values to a value
	var v = 0;

	arr.forEach(function (ele) {
		v = v * base + ele;
	});
	return v;
}

function expand(val, base, length) {	// convert value to array of digit values
	var arr = [];
	var i = length - 1;

	while (i >= 0) {
		arr[i] = val % base;
		val = (val - arr[i]) / base;
		i -= 1;
	}
	return arr;
}

function next(arr, base) {		// calculate the next kaprekar value
	var copy;
	var a;
	var b;

	function order(a, b) {
		return a - b;
	}

	arr.sort(order);		// ascending order
	copy = arr.concat();
	copy.reverse();			// descending order
	b = val(arr, base);
	a = val(copy, base);

	return (a - b);
}

function xprint(val, base, length) {	// print natural numbers in bases 2..36
	var c = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	var arr = expand(val, base, length);
	var s = "";

	arr.forEach(function (ele) {
		s += c[ele];
	});

	return s;
}

function kaprekar(v, base, length) {	// main function
	var arr;
	var map = [];
	var res = "";
	var count = 0;

//	return "kaprekar(" + v.toString(base).toUpperCase() + ", " + base + ", " + length + ")";

	while (map.indexOf(v) < 0) {				// not seen this number before
		res += xprint(v, base, length) + " → ";	// print in appropriate base
		map.push(v);							// save it in the map
		arr = expand(v, base, length);			// expand it to array format
		v = next(arr, base);					// compute next number in the sequence
		count += 1;								// count sequence length
	}
	res += xprint(v, base, length);				// print the final number
	res += " (" + count + ")";					// and sequence length
	return res;
}

/////////////////////////////////////////////////////////////////////////////////////////

function process() {
    var base = parseInt(numberBase.value, 10);
    var length = parseInt(digitsLength.value, 10);
    var v;

   	if (Number.isNaN(base)) {
        calculatedSequence.value = "Invalid Number Base";
        return;
    }
    if ((base < 2) || (base > 36)) {
        calculatedSequence.value = "Number Base Out of Range 2..36";
        return;
    }

    if (Number.isNaN(length)) {
        calculatedSequence.value = "Invalid Number of Digits";
        return;
    }
    if ((length < 2) || (length > 10)) {
        calculatedSequence.value = "Number of Digits Out of Range 2..10";
        return;
    }

    v = parseInt(firstNumber.value, base);

    if (Number.isNaN(v)) {
        calculatedSequence.value = "Invalid First Number";
        return;
    }

    if ((v < 0) || (firstNumber.value.length !== length)) {
        calculatedSequence.value = "First Number Out of Range";
        return;
    }

    calculatedSequence.value = kaprekar(v, base, length);
}

function test_case() {
	numberBase.value = 10;
	digitsLength.value = 4;
	firstNumber.value = 3524;
	calculatedSequence.value = "";
	process();
}

var Return = 13;
var Tab = 9;
var Backspace = 8;

numberBase.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	if (x === Return) {
		process();
    } else {
        calculatedSequence.value = "";
    }
};

digitsLength.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	if (x === Return) {
		process();
    } else {
		calculatedSequence.value = "";
    }
};

firstNumber.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	if (x === Return) {
		process();
    } else {
        calculatedSequence.value = "";
    }
};

calculatedSequence.onkeydown = function (event) {
	var x = event.which || event.keyCode;

 	if (x === Return) {
		process();
	} else if (x === Tab) {
    	event.preventDefault();
		event.stopPropagation();
		numberBase.select();
    } else if (x !== Backspace) {
    	event.preventDefault();
		event.stopPropagation();
    }
};

execButton.onmousedown = function () {
	execButton.style.opacity = "0.5";
};
execButton.onmouseup = function () {
	execButton.style.opacity = "1.0";
	process();
};

gWidgetName = "Kaprekar Sequence Web Widget\n\n";
test_case();
