/*
	Lensmaker - computes using the Lensmaker's Equation
    Copyright © 2019 Harry Whitfield

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.    See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA

    Lensmaker - version 1.0
    6 April, 2019
    Copyright 2019 Harry Whitfield
    mailto:g6auc@arrl.net
 */

/*jslint browser, this */

/*global paramOne, paramTwo, paramThree, paramFour, paramFive,
    headingOne, headingTwo, headingThree, headingFour, headingFive, clearData
*/

/*property
    focus, inRange, isNaN, keyCode, onkeydown, onkeypress, preventDefault,
    prototype, stopPropagation, title, toFixed, value, which
*/

// map custom names onto model names
var rIndex = paramOne;
var radiusOne = paramTwo;
var radiusTwo = paramThree;
var thickness = paramFour;
var opticalPower = paramFive;

var rIndexHeading = headingOne;
var radiusOneHeading = headingTwo;
var radiusTwoHeading = headingThree;
var thicknessHeading = headingFour;
var opticalPowerHeading = headingFive;

var gErrMess = "Invalid data";

var skinPref = false;

Number.prototype.inRange = function (a, b) {
    return (this >= a) && (this <= b);
};

function opticalPwr(n, R1, R2, d) {
    // n is the refractive index of the lens material
    // R1 is the radius of curvature (with sign, see below) of the lens surface closer to the light source
    // R2 is the radius of curvature of the lens surface farther from the light source
    // d is the thickness of the lens (the distance along the lens axis between the two surface vertices)

    // R1 > 0 and R2 < 0 indicate convex surfaces
    // R1 < 0 and R2 > 0 indicate concave surfaces

    // returns the optical power (in diopters) of the lens 1/f, where f is the focal length of the lens

    // radii R1, R2 and distance d to be in meters


    return (n - 1) * (1 / R1 - 1 / R2 + (n - 1) * d / (n * R1 * R2));
}

function process() {
    var n = parseFloat(rIndex.value);
    var R1 = parseFloat(radiusOne.value);
    var R2 = parseFloat(radiusTwo.value);
    var d = parseFloat(thickness.value);
    var power;
    var focalLength;

    if (Number.isNaN(n)) {
        opticalPower.value = "Invalid Refractive Index";
        return;
    }
    if (!n.inRange(0.8, 10)) {
        opticalPower.value = "Refractive Index Out of Range";
        return;
    }

    if (Number.isNaN(R1)) {
        opticalPower.value = "Invalid Radius One Value";
        return;
    }
    if (!(R1.inRange(-20, -0.001) || R1.inRange(0.001, 20))) {
        opticalPower.value = "Radius One Value Out of Range";
        return;
    }

    if (Number.isNaN(R2)) {
        opticalPower.value = "Invalid Radius Two Value";
        return;
    }
    if (!(R2.inRange(-20, -0.001) || R2.inRange(0.001, 20))) {
        opticalPower.value = "Radius Two Value Out of Range";
        return;
    }

    if (Number.isNaN(d)) {
        opticalPower.value = "Invalid Thickness Value";
        return;
    }
    if (!d.inRange(0.0001, 0.5)) {
        opticalPower.value = "Thickness Value Out of Range";
        return;
    }

    power = opticalPwr(n, R1, R2, d);
    opticalPower.value = power.toFixed(6);

    focalLength = 1 / power;

    opticalPower.title = "Focal length: " + focalLength.toFixed(6) + " meters";
}

rIndex.onkeypress = function (event) {
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

rIndex.onkeydown = function (event) {
	var x = event.which || event.keyCode;

	if (x === 9) {
        event.preventDefault();
        event.stopPropagation();
  		radiusOne.focus();
	}
};

radiusOne.onkeypress = function (event) {
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

radiusOne.onkeydown = function (event) {
	var x = event.which || event.keyCode;

	if (x === 9) {
        event.preventDefault();
        event.stopPropagation();
  		radiusTwo.focus();
	}
};

radiusTwo.onkeypress = function (event) {
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

radiusTwo.onkeydown = function (event) {
    "use strict";
	var x = event.which || event.keyCode;

	if (x === 9) {
        event.preventDefault();
        event.stopPropagation();
  		thickness.focus();
	}
};

thickness.onkeypress = function (event) {
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

thickness.onkeydown = function (event) {
    "use strict";
	var x = event.which || event.keyCode;

	if (x === 9) {
        event.preventDefault();
        event.stopPropagation();
        rIndex.focus();

	}
};

opticalPower.onkeydown = function (event) {
    "use strict";
	var x = event.which || event.keyCode;

	if (x === 9) {
        event.preventDefault();
        event.stopPropagation();
	}
};

rIndex.value = "1.5";
radiusOne.value = "1.0";
radiusTwo.value = "-1.0";
thickness.value = "0.005";
process();

rIndex.focus();
