﻿/*
	Lunar Date - A Gregorian to Lunar date converter.
	Copyright © 2004-2015 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
	
	Lunar Date - version 1.9
	31 March, 2008 - updated 24 February, 2015
	Copyright © 2004-2015 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*global sightingDelay */

/*properties
    UTC, day, era, floor, getDate, getFullYear, getMonth, getTime, getUTCDate,
    getUTCFullYear, getUTCMonth, month, round, sqrt, toFixed, year
*/

var lunarYear   = 354.36707;	// days
var lunarMonth  = 29.53058917; 	// days

var jdClockBase = 2440587.5;	// 01/01/1970 00:00:00 UTC
var jdY2K       = 2451544.5;	// 01/01/2000 00:00:00 UTC

var secsPerDay  = 86400;
var mSecsPerDay = 86400000;

var weekDays    = ["Monday   ", "Tuesday  ", "Wednesday", "Thursday ", "Friday   ", "Saturday ", "Sunday   "];

var HijriDays   = ["al-Ithnain", "al-Thalathaa", "al-Arbia'aa", "al-Khamis", "al-Jum'a", "as-Sabt", "al-Ahad"];

var theMonths   = ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"];

var lunarMonths = ["Muharram", "Safar", "Rabi' al-awwal", "Rabi' al-thani", "Jumada al-awwal", "Jumada al-thani", "Rajab", "Sha'ban", "Ramadan", "Shawwal", "Dhu al-Qi'dah", "Dhu al-Hijjah"];

function weekDay(jd) {
	return weekDays[Math.floor(jd) % 7];
}

function hijriDay(jd) {
	return HijriDays[Math.floor(jd) % 7];
}

function CalDate(day, month, year, era) {
	this.day   = day;
	this.month = month;
	this.year  = year;
	this.era   = era;
}

function today() {
	var d  = new Date();
	return new CalDate(d.getDate(), d.getMonth() + 1, d.getFullYear(), 'CE');
}

function fracpt(x) { return x - Math.floor(x); }

function leapYear(year) {
	if ((year %   4) !== 0) { return false; }
	if ((year % 400) === 0) { return true; }
	if ((year % 100) === 0) { return false; }
	return true;
}

function getDaysIn(month, year) {	// month 1..12
	var daysInMonth = [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
	
	if (month !== 2) { return daysInMonth[month - 1]; }
	if (leapYear(year)) { return 29; }
	return 28;
}

function getJulianDay(era, y, m, d, h, mn, s) {	// days from noon UTC on 1 Jan 4713 BCE
	var date;
	
	if ((era !== "CE") || (y < 1970)) { return NaN; }
	date = Date.UTC(y, m - 1, d, h, mn, s);
	return (jdClockBase + ((Math.floor(date / 1000)) / secsPerDay));
}
    
function jdNow() {			// days from noon UTC on 1 Jan 4713 BCE
	var d = new Date();
	
	return (2415020.5 + ((2208988800 + Math.floor(d.getTime() / 1000)) / 86400)).toFixed(6);
}

function lunarToGregorian(date) {
	var year = date.year,
		month = date.month,
		day = date.day,
//		era = date.era,
		hdays = lunarYear * year + lunarMonth * (month - 1) + day - 1,
		bdays = lunarYear * 1420 + lunarMonth * (9     - 1) +  24 - 1,
		ms,
		d,
		cyear,
		cmonth,
		cday;
	
	hdays = hdays - bdays + jdY2K - jdClockBase + 0.25;
		
	ms = Math.floor(mSecsPerDay * hdays);
	d = new Date(ms);
	cyear  = d.getUTCFullYear();
	cmonth = d.getUTCMonth() + 1;
	cday   = d.getUTCDate();
	return new CalDate(cday, cmonth, cyear, "CE");
}

function newMoon(jd) {
	// D = 5.597661 + 29.5305888610*N + 102.026E-12)*N*N
	// where D is the number of days (and fractions) since 2000-01-01 00:00:00 (JD2451544.5), and N is an integer.
	var D = jd - 2451544.5,
		a = 102.026E-12,
		b = 29.5305888610,
		c = 5.597661 - D,
		N = Math.floor((-b + Math.sqrt(b * b - 4 * a * c)) / (2 * a));
	
	return D - (5.597661 + b * N + a * N * N);					// zero-based day (and fraction) in month from new moon
}

function gregorianToLunar(date) {
	var year = date.year,
		month = date.month,
		day = date.day,
		era = date.era,
		jd = getJulianDay(era, year, month, day, 12, 0, 0),	// noon on given date
		hdays = jd - 1948440,
		hyears  = hdays / lunarYear,
		cyear = Math.floor(hyears) + 1,
		hmonths,
		cmonth,
		cday,
		lday;
	
	hdays = lunarYear * fracpt(hyears);
	
	hmonths = hdays / lunarMonth;
	cmonth = Math.floor(hmonths) + 1;
	hdays = lunarMonth * fracpt(hmonths);
	cday = Math.round(hdays) + 1;
	lday = Math.floor(newMoon(jd - sightingDelay)) + 1;
	if ((lday !== cday)) {
		// print('cday=' + cday + ' lday' + lday);
		if ((cday < 3) && (lday > 28)) {
			cmonth -= 1;
			if (cmonth === 0) {
				cmonth = 12;
				cyear -= 1;
			}
		} else if ((cday > 28) && (lday <  3)) {
			cmonth += 1;
			if (cmonth === 13) {
				cmonth =  1;
				cyear += 1;
			}
		}
	}
	cday = lday;
	return new CalDate(cday, cmonth, cyear, "AH");
}

function convertDate(date) {
	var year  = date.year,
		month  = date.month,
		day  = date.day,
		era  = date.era,
		invalidDate = new CalDate(0, 0, 0, ""),
		lunar,
		maxDay;
	
	day   = parseInt(day, 10);
	if (isNaN(day)) { return invalidDate; }
	month = parseInt(month, 10);
	if (isNaN(month)) { return invalidDate; }
	year  = parseInt(year, 10);
	if (isNaN(year)) { return invalidDate; }
	if ((era !== "CE") && (era !== "AH")) { return invalidDate; }
	lunar = (era === "AH");
	if ((lunar && (year < 1390)) || (!lunar && (year < 1970))) { return invalidDate; }
	maxDay = 31;
	if (lunar) { maxDay = 30; }
	if (year  >   9999) { return invalidDate; }
	if ((month < 1) || (month >     12)) { return invalidDate; }
	if ((day   < 1) || (day   > maxDay)) { return invalidDate; }
	if (!lunar) { if (day > getDaysIn(month, year)) { return invalidDate; } }
	date = new CalDate(day, month, year, era);
	if (lunar) {return lunarToGregorian(date); }
	return gregorianToLunar(date);
}

function monthLen(jd) {
	// D = 5.597661 + 29.5305888610*N + 102.026E-12)*N*N
	// where D is the number of days (and fractions) since 2000-01-01 00:00:00 (JD2451544.5), and N is an integer.
	var D = jd - 2451544.5,
		a = 102.026E-12,
		b = 29.5305888610,
		c = 5.597661 - D,
		N = Math.floor((-b + Math.sqrt(b * b - 4 * a * c)) / (2 * a));
	
	return b + a * (2 * N + 1);
}
