/*
	Morse Key - browser version 1.1
	17 June, 2015
	Copyright 2012-2015 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*
    ProSign     Pattern     Key         Meaning

    HH          ........    ^ or BS     Error
    VE          ...-.       [           Start of transmission
    CT/KA       _._._       <           Start of message
    AR          .-.-.       >           End of message
    SK/VA       ..._._      ]           End of transmission, End of work
    AS          .-...       |           Stand by, Wait
    BT          -...-       =           Separator
*/

/*jslint browser, multivar */

/*global drawImage, pointer, pointerImage, append, playSound, alert, setTimeout,
    clearTimeout
*/

/*properties
    ' ', '!', '"', '#', $, '&', '\'', '(', ')', '+', ',', '-', '.', '/', '0',
    '1', '2', '3', '4', '5', '6', '7', '8', '9', ':', ';', '<', '=', '>', '?',
    '@', A, B, C, D, E, F, G, H, I, J, K, L, M, N, O, P, Q, R, S, T, U, V, W, X,
    Y, Z, '[', ']', '^', _, init, interval, length, onTimerFired, replace, round,
    stop, ticking, timeoutID, toUpperCase, update, '|'
*/

var morse = (function () {
    "use strict";
	var morseCode = {" ": "#",
	            A: ".-", B: "-...", C: "-.-.", D: "-..", E: ".", F: "..-.", G: "--.", H: "....", I: "..",
				J: ".---", K: "-.-", L: ".-..", M: "--", N: "-.", O: "---", P: ".--.", Q: "--.-", R: ".-.",
				S: "...", T: "-", U: "..-", V: "...-", W: ".--", X: "-..-", Y: "-.--", Z: "--..",
				"1": ".----", "2": "..---", "3": "...--", "4": "....-", "5": ".....",
				"6": "-....", "7": "--...", "8": "---..", "9": "----.", "0": "-----",
				".": "#+.-.-.-#", ",": "#+--..--#", "?": "#+..--..#",
				"'": ".----.", "!": "-.-.--", "/": "-..-.", "(": "-.--.", ")": "-.--.-", "&": ".-...",
				":": "---...", ";": "-.-.-.", "=": "-...-", "+": ".-.-.", "-": "-....-", "_": "..--.-", "\"": ".-..-.",
				"$": "...-..-", "@": ".--.-.",
				"^": "........#",	// HH Error
				"[": "...-.#",		// VE Start of transmission
				"<": "-.-.-#",		// CT/KA - Start of message
				">": ".-.-.#",		// AR - End of message
				"]": "...-.-#",		// SK/VA - End of work
				"|": ".-...#"},		// AS Stand by, Wait

		interval = {".": 0.120, "-": 0.240, "+": 0.120, "#": 0.240},
//		sound    = {".": "dot120.mp3", "-": "dash240.mp3"},

		buffer = [],
		on = 0,
		off = 0,
		bufLen = 1024,
		pattern = "",
		idx = 0,
		charge = 0,
		time = 0,

		getc = function () {
			var c;

			if (on === off) {
			    return null;
			}
			c = buffer[off];
			off = (off + 1) % bufLen;
			append(c
				.replace(/\^/, "HH ")
				.replace(/\[/, "VE ")
				.replace(/</, "CT ")
				.replace(/>/, "AR ")
				.replace(/\]/, "SK ")
				.replace(/\|/, "AS ")
				.replace(/\=/, "BT "));
			return c.toUpperCase();
		},

		get = function () {
			var c, token;

			if (pattern !== "") {
				token = pattern[idx];
				idx += 1;
				if (idx >= pattern.length) {
				    pattern = "";
				}
				return token;
			}
			c = getc();
			if (c === null) {
			    return null;
			}
			pattern = morseCode[c] + "+";
			idx = 1;
			return pattern[0];
		},

		dot,
		dash,

		timer = {},

		putc = function (c, delay) {
			var i, nextOn;

			i = 0;
			while (i < c.length) {
				nextOn = (on + 1) % bufLen;
				if (nextOn === off) {
					alert("Input buffer is full.");
					return;
				}
				if ((on === off) && (pattern === "")) {
					timer.interval = delay;
					timer.ticking = true;
					timer.timeoutID = setTimeout(timer.onTimerFired, 1000 * timer.interval);
				}
				buffer[on] = c[i];
				on = nextOn;
				i += 1;
			}
		},

		setMeter = function (mA) {	// was , duration
			var rotation = Math.round(3.4 * mA - 33);

			drawImage(pointer, pointerImage, 0, 0, rotation);
		},

		delta = 0.005,
		itself;

	timer.ticking = false;

	timer.onTimerFired = function () {
		var c, duration;

		timer.ticking = false;
		c = get();
		if (c === null) {
			charge = 0;
			time = 0;
			duration = 240;
			setMeter(0, duration);
			return;
		}
		timer.interval = interval[c] + delta;	// was minus
		if (c === ".") {
		    playSound(dot);
		}
		if (c === "-") {
		    playSound(dash);
		}

		timer.ticking = true;
		timer.timeoutID = setTimeout(timer.onTimerFired, 1000 * timer.interval);

		if (c === ".") {
			charge += 1;
			time += 2;
			duration = 120;
		} else if (c === "-") {
			charge += 3;
			time += 4;
			duration = 240;
		} else if (c === "+") {
			time += 2;
			duration = 1000 * interval["+"];
		} else {
			charge = 0;
			time = 0.001;
			duration = 1000 * interval["#"];
		}
		setMeter(10 * charge / time, duration);
	};

	itself = function (data, delay) {
		var i, c;

		data = data.replace(/\r\n?|\n/g, " ");
		delay = delay || 0.01;

		i = 0;
		while (i < data.length) {
			c = data[i];
			if (morseCode[c.toUpperCase()] !== undefined) {
				putc(c, delay);
			} else {
				alert("Invalid character '" + c + "'");
				return;
			}
			i += 1;
		}
	};

	itself.stop = function () {
		var duration;

		timer.ticking = false;
		clearTimeout(timer.timeoutID);
		buffer = [];
		on = 0;
		off = 0;
		bufLen = 1024;
		pattern = "";
		idx = 0;
		charge = 0;
		time = 0;
		duration = 240;
		setMeter(0, duration);
		//alert("STOPPED");
	};

	itself.update = function (speedPref) {	// 5 to 20
		var speed = (60 / speedPref - 2.16) / 7;

		interval["+"] = speed;
		interval["#"] = 2 * speed;

		return speedPref;
	};

	itself.init = function (dotBuffer, dashBuffer) {
		dot = dotBuffer;
		dash = dashBuffer;
	};

	return itself;
}());
