/*
	Roman Numerals

	This program is free software; you can redistribute it and/or modify it under
	the terms of the GNU General Public License as published by the Free Software
	Foundation; either version 2 of the License, or (at your option) any later
	version.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY
	WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
	PARTICULAR PURPOSE.	 See the GNU General Public License for more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
	St, Fifth Floor, Boston, MA	 02110-1301	 USA

	Roman Numerals - version 1.1
	8 April, 2016
	Copyright 2016 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*jslint browser, devel */

/*property
    $, C, D, I, L, M, S, V, X, every, forEach, isNaN, join, match, replace,
    split, toUpperCase
*/

/*
function min(a, b) {	// returns smaller of two numbers
	"use strict";

	return (
		a < b
		? a
		: b
	);
}
*/

/*
Fractions:
»		1/1728	siliqua
℈		1/288	scripulum
$		1/144	dimidia sextula 𐆔
£		1/24	semuncia

●		1/12	uncia
●●		1/6		sextans
●●●		1/4		quadrans
●●●●	1/3		triens
●●●●●	5/12	quincunx
S		1/2		semis
S●		7/12	septunx
S●●		2/3		bes
S●●●	3/4		dodrans
S●●●●	5/6		dextans
S●●●●●	11/12	deinx
I		1		as
*/

var OZ = 1 / 12;
var SU = 1 / 24;
var DS = 1 / 144;
var SC = 1 / 288;
var SI = 1 / 1728;
var SSI = 1 / 3456;

var lookForRoman = (
	"^([[M]]){0,3}" +
	"([[C]][[M]]|[[C]][[D]]|([[D]])?([[C]]){0,3})" +
	"([[X]][[C]]|[[X]][[L]]|([[L]])?([[X]]){0,3})" +
	"([M][[X]]|[M][[V]]|([[V]])?([M]){0,3})" +
	"([C][M]|[C][D]|([D])?([C]){0,3})" +
	"([X][C]|[X][L]|([L])?([X]){0,3})" +
	"(M[X]|M[V]|([V])?M{0,3})" +
	"(CM|CD|D?C{0,3})" +
	"(XC|XL|L?X{0,3})" +
	"(IX|IV|V?I{0,3})" +
	"(S?●{0,5})" +
	"(£?\\${0,5})" +
	"(℈?»{0,5})$"
);

lookForRoman = lookForRoman.replace(/\[/g, "\\(").replace(/\]/g, "\\)");
lookForRoman = new RegExp(lookForRoman, "");

function goodRoman(r) { // checks Roman number is in compact form and in (0..4e9)
	"use strict";

	return (r.match(lookForRoman) !== null) && (r !== "");
}

function compact(r) {	// converts a Roman number from full to compact form
	"use strict";

	r = r.replace("((D))((C))((C))((C))((C))", "((C))((M))");
	r = r.replace("((C))((C))((C))((C))", "((C))((D))");
	r = r.replace("((L))((X))((X))((X))((X))", "((X))((C))");
	r = r.replace("((X))((X))((X))((X))", "((X))((L))");
	r = r.replace("((V))(M)(M)(M)(M)", "(M)((X))");
	r = r.replace("(M)(M)(M)(M)", "(M)((V))");

	r = r.replace("(D)(C)(C)(C)(C)", "(C)(M)");
	r = r.replace("(C)(C)(C)(C)", "(C)(D)");
	r = r.replace("(L)(X)(X)(X)(X)", "(X)(C)");
	r = r.replace("(X)(X)(X)(X)", "(X)(L)");
	r = r.replace("(V)MMMM", "M(X)");
	r = r.replace("MMMM", "M(V)");

	r = r.replace("DCCCC", "CM");
	r = r.replace("CCCC", "CD");
	r = r.replace("LXXXX", "XC");
	r = r.replace("XXXX", "XL");
	r = r.replace("VIIII", "IX");
	r = r.replace("IIII", "IV");

	return r;
}

function uncompact(r) { // converts a Roman number from compact to full form
	"use strict";

	r = r.replace("IV", "IIII");
	r = r.replace("IX", "VIIII");
	r = r.replace("XL", "XXXX");
	r = r.replace("XC", "LXXXX");
	r = r.replace("CD", "CCCC");
	r = r.replace("CM", "DCCCC");

	r = r.replace("M(V)", "MMMM");
	r = r.replace("M(X)", "(V)MMMM");
	r = r.replace("(X)(L)", "(X)(X)(X)(X)");
	r = r.replace("(X)(C)", "(L)(X)(X)(X)(X)");
	r = r.replace("(C)(D)", "(C)(C)(C)(C)");
	r = r.replace("(C)(M)", "(D)(C)(C)(C)(C)");

	r = r.replace("(M)((V))", "(M)(M)(M)(M)");
	r = r.replace("(M)((X))", "((V))(M)(M)(M)(M)");
	r = r.replace("((X))((L))", "((X))((X))((X))((X))");
	r = r.replace("((X))((C))", "((L))((X))((X))((X))((X))");
	r = r.replace("((C))((D))", "((C))((C))((C))((C))");
	r = r.replace("((C))((M))", "((D))((C))((C))((C))((C))");

	return r;
}

function toRoman(n) {	// converts a decimal number into a compact Roman number
	"use strict";		// returns null if n is not a number
						// n must be less than 4e9 in magnitude

	var decimals = [
		1e9, 5e8, 1e8, 5e7, 1e7, 5e6, 1e6, 5e5, 1e5, 5e4, 1e4, 5e3,
		1000, 500, 100, 50, 10, 5, 1, 0.5, OZ, SU, DS, SC, SI
	];
	var roman = [
		"((M))", "((D))", "((C))", "((L))", "((X))", "((V))", "(M)", "(D)", "(C)", "(L)", "(X)", "(V)",
		"M", "D", "C", "L", "X", "V", "I", "S", "●", "£", "$", "℈", "»"
	];
	var r = "";
	var sign = "";

	if (Number.isNaN(n)) {
		return null;
	}
	if (n === 0) {
		return "";
	}
	if (n < 0) {
		sign = "-";
		n = -n;
	}
	if (n >= 4e9) {
		return "Number is very large!\nMagnitude >= 4e9.";
	}

	if (n < SSI) {
		return "Number is very small!\nMagnitude < 1/3456";
	}

	n += SSI; // rounding

	decimals.forEach(function (ele, i) {
		while (n >= ele) {
			r += roman[i];
			n -= ele;
		}
	});

	return sign + compact(r);
}

function splitRoman(r) {	// r is a string of (unsorted) Roman numerals
	"use strict";			// returns an array with r split into its separate elements
							// if an error occurs, returns the position of the error
	var a;
	var lookFor = /\(\([MDCLXV]\)\)|\([MDCLXV]\)|[MDCLXVIS●£$℈»]/g;
	var errPos = 0;
	var b;

	if (r === "") {
		return errPos;
	}

	a = r.match(lookFor);
	if (a === null) {
		return errPos;
	}

	b = a.join("");

	if (b !== r) {	// if there have been errors, find the error position
		r.split("").every(function (ele, i) {
			errPos = i;
			return (ele === b[i]);
		});
		return errPos;
	}
	return a;
}

function fromRoman(r, check) {		// converts a [compact] Roman number into a decimal number
	"use strict";					// if check is true,
									// returns null if r is not compact or not in (0..40000000)

//	var val = {"M": 1000, "D": 500, "C": 100, "L": 50, "X": 10, "V": 5, "I": 1, "S": 0.5, "●": OZ, "£": SU, "$": DS, "℈": SC, "»": SI};
	var val = {"((M))": 1e9, "((D))": 5e8, "((C))": 1e8, "((L))": 5e7, "((X))": 1e7, "((V))": 5e6, "(M)": 1e6, "(D)": 5e5, "(C)": 1e5, "(L)": 5e4, "(X)": 1e4, "(V)": 5e3, "M": 1000, "D": 500, "C": 100, "L": 50, "X": 10, "V": 5, "I": 1, "S": 0.5, "●": OZ, "£": SU, "$": DS, "℈": SC, "»": SI};

	var n = 0;

	r = r.toUpperCase();
	if (check && !goodRoman(r)) {
		return null;
	}

	r = uncompact(r);
	r = splitRoman(r);

	if (typeof r === "number") {
		alert("Error at position " + r);
		return NaN;
	}

	r.forEach(function (ele) {
		n += val[ele];
	});
	return n;
}

//////////////////////////////////////////////////////////////////////////////////////////

/*
function compress(r) {	// converts runs of smaller valued numerals into larger valued numerals
	"use strict";

	return r
		.replace(/\$\$\$\$\$\$/g, "£")
		.replace(/££/g, "●")
		.replace(/●●●●●●/g, "S")
		.replace(/SS/g, "I")

		.replace(/IIIII/g, "V")
		.replace(/VV/g, "X")
		.replace(/XXXXX/g, "L")
		.replace(/LL/g, "C")
		.replace(/CCCCC/g, "D")
		.replace(/DD/g, "M")
		.replace(/MMMMM/g, "(V)")

		.replace(/\(V\)\(V\)/g, "(X)")
		.replace(/\(X\)\(X\)\(X\)\(X\)\(X\)/g, "(L)")
		.replace(/\(L\)\(L\)/g, "(C)")
		.replace(/\(C\)\(C\)\(C\)\(C\)\(C\)/g, "(D)")
		.replace(/\(D\)\(D\)/g, "(M)")
		.replace(/\(M\)\(M\)\(M\)\(M\)\(M\)/g, "((V))")

		.replace(/\(\(V\)\)\(\(V\)\)/g, "((X))")
		.replace(/\(\(X\)\)\(\(X\)\)\(\(X\)\)\(\(X\)\)\(\(X\)\)/g, "((L))")
		.replace(/\(\(L\)\)\(\(L\)\)/g, "((C))")
		.replace(/\(\(C\)\)\(\(C\)\)\(\(C\)\)\(\(C\)\)\(\(C\)\)/g, "((D))")
		.replace(/\(\(D\)\)\(\(D\)\)/g, "((M))");
}

function addRoman(a, b, check) {	// add Roman integers by converting to and from decimal
	"use strict";					// if check is true, returns null or undefined if
									// a or b is not compact or not in [1..3999]
	a = fromRoman(a, check);
	if (a === null) {
		return null;
	}
	b = fromRoman(b, check);
	if (b === null) {
		return undefined;
	}

	return toRoman(a + b);
}

function addRoman(a, b, check) {	// add Roman numbers without converting to and from decimal
	"use strict";					// if check is true, returns null or undefined if
									// a or b is not compact or not in (0..4000)
	var order = function (a, b) {
		var val = {"M": 1000, "D": 500, "C": 100, "L": 50, "X": 10, "V": 5, "I": 1, "S": 0.5, "●": OZ, "£": SU, "$": DS};
		return val[b] - val[a];
	};
	var r;

	a = a.toUpperCase();
	b = b.toUpperCase();
	if (check && !goodRoman(a)) {
		return null;
	}
	if (check && !goodRoman(b)) {
		return undefined;
	}

	r = uncompact(a) + uncompact(b);		// uncompact both then concatenate
	r = r.split("").sort(order).join("");	// sort into descending order
	r = compress(r);						// compress
	return compact(r);						// compact
}

function compareRoman(a, b, check) {	// compare Roman integers by converting to and from decimal
	"use strict";						// returns >0 if a > b, <0 if a < b, 0 if a = b
										// if check is true, returns null or undefined if
										// a or b is not compact or not in [1..3999]
	a = fromRoman(a, check);
	if (a === null) {
		return null;
	}
	b = fromRoman(b, check);
	if (b === null) {
		return undefined;
	}

	return a - b;
}

function compareRoman(a, b, check) {	// compare Roman numbers without converting to and from decimal
	"use strict";						// returns >0 if a > b, <0 if a < b, 0 if a = b
										// if check is true, returns null or undefined if
										// a or b is not compact or not in (0..4000)

	var val = {"M": 1000, "D": 500, "C": 100, "L": 50, "X": 10, "V": 5, "I": 1, "S": 0.5, "●": OZ, "£": SU, "$": DS};
	var len;
	var o;
	var i = 0;

	a = a.toUpperCase();
	b = b.toUpperCase();
	if (check && !goodRoman(a)) {
		return null;
	}
	if (check && !goodRoman(b)) {
		return undefined;
	}

	a = uncompact(a);					// uncompact both
	b = uncompact(b);
	len = min(a.length, b.length);		// length in common

	while (i < len) {
		o = val[a[i]] - val[b[i]];
		if (o !== 0) {					// values differ
			return o;
		}
		i += 1;
	}
	return a.length - b.length;			// if common part equal
}

function subtractRoman(a, b, check) {	// subtract Roman integers by converting to and from decimal
	"use strict";						// if check is true, returns null or undefined if
										// a or b is not compact or not in [1..3999]
										// returns "" if a and b are equal
										// returns "ERROR" if a < b

	a = fromRoman(a, check);
	if (a === null) {
		return null;
	}
	b = fromRoman(b, check);
	if (b === null) {
		return undefined;
	}
	if (a === b) {
		return "";
	}
	if (a < b) {
		return "ERROR";
	}

	return toRoman(a - b);
}

function subtractRoman(a, b, check) {	// subtract Roman numbers without converting to and from decimal
	"use strict";						// if check is true, returns null or undefined if
										// a or b is not compact or not in (0..4000)
										// returns "" if a and b are equal
										// returns "ERROR" if a < b

	var convert = function (a) {		// convert full Roman number to counter array
		var idx = {"M": 0, "D": 1, "C": 2, "L": 3, "X": 4, "V": 5, "I": 6, "S": 7, "●": 8, "£": 9, "$": 10};
		var c = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]; // counters for each numeral

		a.split("").forEach(function (ele) {
			c[idx[ele]] += 1;
		});
		return c;
	};

	var subtract = function (a, b) {	// subtract counter arrays
		var c = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];

		a.forEach(function (ele, i) {
			c[i] = ele - b[i];
		});
		return c;
	};

	var reconvert = function (c) {		// convert counter array to full Roman number
		var keys = ["M", "D", "C", "L", "X", "V", "I", "S", "●", "£", "$"];
		var k;
		var r = "";

		keys.forEach(function (ele, j) {
			k = 0;
			while (k < c[j]) {
				r += ele;
				k += 1;
			}
		});
		return r;
	};

	var c;											// difference array set below
	var borrow = [6, 2, 6, 2, 5, 2, 5, 2, 5, 2];	// at indexes 10 to 1 in difference array c
	var com = compareRoman(a, b, check);

	if ((com === null) || (com === undefined)) {
		return com;
	}
	if (com === 0) {
		return "";
	}
	if (com < 0) {
		return "ERROR";
	}

	a = a.toUpperCase();
	b = b.toUpperCase();
	a = uncompact(a);					// uncompact both
	b = uncompact(b);
	a = convert(a);						// make counter arrays
	b = convert(b);
	c = subtract(a, b);					// make difference array

	borrow.forEach(function (ele, i) {	// propagate any borrows
		var j = 10 - i;

		if (c[j] < 0) {		// we need to borrow
			c[j] += ele;	// add the borrowed numerals to this numeral's count
			c[j - 1] -= 1;	// take 1 off the count of next higher valued numeral
		}
	});

	c = reconvert(c);					// convert back to full format Roman number
	return compact(c);
}

function multiplyRoman(a, b, check) {	// multiply Roman integers by converting to and from decimal
	"use strict";						// if check is true, returns null or undefined if
										// a or b is not compact or not in [1..3999]
	a = fromRoman(a, check);
	if (a === null) {
		return null;
	}
	b = fromRoman(b, check);
	if (b === null) {
		return undefined;
	}

	return toRoman(a * b);
}

function multiplyRoman(a, b, check) {	// multiply Roman numbers without converting to and from decimal
	"use strict";						// if check is true, returns null or undefined if
										// a or b is not compact or not in (0..4000)
	var prod = [	// Roman Multiplication Table - must be in full form
		["$", "£", "●", "●+●+●+●+●", "S+●+●+●+●", "I+I+I+I+●+●", "V+I+I+I+●+●+●+●", "X+X+X+X+I+S+●+●", "L+X+X+X+I+I+I+●+●+●+●"],	// times ● = 1/12
		["£", "●+●+●", "S", "I+I+S", "V", "X+X+V", "L", "C+C+L", "D"],									// time S = 0.5
		["●", "S", "I", "V", "X", "L", "C", "D", "M"],													// times I = 1
		["●+●+●+●+●", "I+I+S", "V", "X+X+V", "L", "C+C+L", "D", "M+M+D", "(V)"],						// times V = 5
		["S+●+●+●+●", "V", "X", "L", "C", "D", "M", "(V)", "(X)"],										// times X = 10
		["I+I+I+I+●+●", "X+X+V", "L", "C+C+L", "D", "M+M+D", "(V)", "(X)+(X)+(V)", "(L)"],				// times L = 50
		["V+I+I+I+●+●+●+●", "L", "C", "D", "M", "(V)", "(X)", "(L)", "(C)"],							// times C = 100
		["X+X+X+X+I+S+●+●", "C+C+L", "D", "M+M+D", "(V)", "(X)+(X)+(V)", "(L)", "(C)+(C)+(L)", "(D)"],	// times D = 500
		["L+X+X+X+I+I+I+●+●+●+●", "D", "M", "(V)", "(X)", "(L)", "(C)", "(D)", "(M)"]					// time M + 1000
	];
	var idx = {"M": 8, "D": 7, "C": 6, "L": 5, "X": 4, "V": 3, "I": 2, "S": 1, "●": 0};
	var order = function (a, b) {
		var val = {"((X))": 1e7, "((V))": 5e6,
				"(M)": 1e6, "(D)": 5e5, "(C)": 1e5, "(L)": 5e4, "(X)": 1e4, "(V)": 5e3,
				"M": 1000, "D": 500, "C": 100, "L": 50, "X": 10, "V": 5, "I": 1,
				"S": 0.5, "●": OZ, "£": SU, "$": DS};
		return val[b] - val[a];
	};
	var multiply = function (a, b) {	// using Roman Multiplication Table
		var p = "";
		a.split("").forEach(function (eleA) {
			b.split("").forEach(function (eleB) {
				if (p !== "") {
					p += "+";
				}
				p += prod[idx[eleA]][idx[eleB]];
			});
		});
		return p;
	};
	var r;

	a = a.toUpperCase();
	b = b.toUpperCase();
	if (check && !goodRoman(a)) {
		return null;
	}
	if (check && !goodRoman(b)) {
		return undefined;
	}

	a = uncompact(a);						// uncompact both
	b = uncompact(b);

	r = multiply(a, b);						// raw product

	r = r.split("+").sort(order).join("");	// sort into descending order
	r = compress(r);						// compress
	return compact(r);						// compact
}

function divideRoman(a, b, check) {		// divide Roman integers by converting to and from decimal
	"use strict";						// if check is true, returns null or undefined if
										// a or b is not compact or not in [1..3999]
	var q;	// quotient
	var r;	// remainder

	a = fromRoman(a, check);
	if (a === null) {
		return null;
	}
	b = fromRoman(b, check);
	if (b === null) {
		return undefined;
	}

	q = Math.floor(a / b);
	r = a - q * b;

	return [toRoman(q), toRoman(r)];
}

function divideRoman(a, b, check) {		// divide Roman numbers without converting to and from decimal
	"use strict";						// if check is true, returns null or undefined if
										// a or b is not compact or not in (0..4000)
	var divide = function (a, b) {
		var roman = ["M", "D", "C", "L", "X", "V", "I", "S", "●"];
		var mults = []; // multiples of the divisor
		var q = "";		// quotient

		roman.forEach(function (ele) {	// form multiples of the divisor
			var p = multiplyRoman(ele, b, false);
			var com = compareRoman(a, p, false);

			if ((com === undefined) || (com < 0)) {
				mults.push("");
			} else {
				mults.push(p);
			}
		});

		mults.forEach(function (ele, i) {
			if (ele !== "") {
				while (compareRoman(a, ele, false) >= 0) {
					q += roman[i];
					a = subtractRoman(a, ele, false);
				}
			}
		});

		return [q, a];
	};
	var d;

	a = a.toUpperCase();
	b = b.toUpperCase();
	if (check && !goodRoman(a)) {
		return null;
	}
	if (check && !goodRoman(b)) {
		return undefined;
	}

	d = divide(a, b);						// raw quotient [q, r]
	return [compact(d[0]), compact(d[1])];	// compact
}
*/
//////////////////////////////////////////////////////////////////////////////////////////

export {fromRoman, goodRoman, splitRoman, toRoman, uncompact};