/*
	New Abacus
	Copyright © 2020 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA	 02110-1301	 USA

	New Abacus - browser version 1.3
	19 june, 2020
	Copyright © 2020 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*jslint browser, devel, for, this */

/*property
    $, C, D, I, L, M, S, V, X, altKey, body, checked, clientX, clientY, div,
    exponent, floor, forEach, fraction, hOffset, index, indexOf, isNaN, keyCode,
    lastIndexOf, length, onchange, onclick, onkeydown, onmousedown, onmouseup,
    opacity, open, pow, removeChild, scrollX, scrollY, setAttribute, shiftKey,
    sign, style, substring, title, toFixed, toPrecision,
    toUpperCase, vOffset, value, which
*/

import {left, newImage, moveObj, newInput, vtop} from "./webWidget.js";
import {goodRoman, splitRoman, toRoman, uncompact} from "./roman.js";

/////////////////////////////////////////// GUI //////////////////////////////////////////

//	newImage(hOffset, vOffset, width, height, src, zOrder, opacity)
//	newInput(hOffset, vOffset, width, height, value, zOrder, style, id)

var base = "Resources/Images/";

var backing = newImage(0, 0, 530, 900, base + "abacusBgBg.png", 0, 1.0);
backing.title = "";

var background = newImage(0, 0, 530, 900, base + "abacusBg.png", 1, 1.0);
background.title = "";

var foreground = newImage(0, 0, 530, 900, base + "bg.png", 2, 1.0);
foreground.title = "";

var style1 = "font-family:'Lucida Grande';font-weight:normal;font-size:24px;text-align:center;color:black;background-color:transparent;";
var style2 = "font-family:'Lucida Grande';font-weight:normal;font-size:24px;text-align:center;color:red;background-color:transparent;";

var extra = "\n\nUse with the alt-key to add the number to the accumulator or with the shift-key to subtract the number.";

var input = newInput(69, 7, 320, 28, "", 3, style1);
input.title = "ACCUMULATOR: Enter a number, then press RETURN.";

var output = newInput(69, 43, 320, 28, "", 3, style2);
output.title = "INPUT: Enter a number, then press RETURN." + extra;

var diags = newInput(69, 851, 320, 28, "Press blue balloon for help!", 3, style2);

//var helpButton = newInput(390, 85, 64, 24, "help info", 3);
//helpButton.setAttribute("type", "button");

var helpButton = newImage(395, 85, 60, 24, base + "help-info.png", 3, 1.0);
helpButton.title = "Displays help information about the abacus.";

//helpButton.onmousedown = function () {
//	helpButton.style.opacity = "0.5";
//};

helpButton.onclick = function () {
//	helpButton.style.opacity = "1.0";
	window.open("Help.html");
};

var compactBox = newInput(72, 85, 8, 24, "compact", 3);
compactBox.setAttribute("type", "checkbox");
compactBox.title = "Enable/disable compact pebble placement.";
compactBox.checked = true;

//var compactButton = newInput(3, 85, 64, 24, "compact", 3);
//compactButton.setAttribute("type", "button");

var compactButton = newImage(5, 85, 60, 24, base + "compact.png", 3, 1.0);
compactButton.title = "Enable/disable compact pebble placement.";

//var splayButton = newInput(208, 85, 46, 24, "<- ->", 3);
//splayButton.setAttribute("type", "button");

var splayButton = newImage(208, 85, 48, 24, base + "splay.png", 3, 1.0);
splayButton.title = "Click to move centered pebbles to the edges.";

var altKey = newInput(360, 85, 8, 24, "alt-key", 3);
altKey.setAttribute("type", "checkbox");
altKey.title = "Enable/disable the on-screen alt-key.";

//var altButton = newInput(288, 85, 64, 24, "alt-key", 3);
//altButton.setAttribute("type", "button");

var altButton = newImage(294, 85, 60, 24, base + "alt-key.png", 3, 1.0);
altButton.title = "Enable/disable the on-screen alt-key.";

var shiftKey = newInput(90, 85, 8, 24, "shift-key", 3);
shiftKey.setAttribute("type", "checkbox");
shiftKey.title = "Enable/disable the on-screen shift-key.";

//var shiftButton = newInput(110, 85, 64, 24, "shift-key", 3);
//shiftButton.setAttribute("type", "button");

var shiftButton = newImage(110, 85, 60, 24, base + "shift-key.png", 3, 1.0);
shiftButton.title = "Enable/disable the on-screen shift-key.";

compactButton.onclick = function () {
	compactBox.checked = !compactBox.checked;
};

altButton.onclick = function () {
	altKey.checked = !altKey.checked;
	if (altKey.checked) {
		shiftKey.checked = false;
	}
};

altKey.onchange = function () {
	if (altKey.checked) {
		shiftKey.checked = false;
	}
};

shiftButton.onclick = function () {
	shiftKey.checked = !shiftKey.checked;
	if (shiftKey.checked) {
		altKey.checked = false;
	}
};

shiftKey.onchange = function () {
	if (shiftKey.checked) {
		altKey.checked = false;
	}
};

//////////////////////////////////////////////////////////////////////////////////////////

var lastRow = 24;
var lastRowExp = 4;
var unitRow = 6;
var maxPower = lastRowExp / 2;
var minPower = lastRow / 2;
var lineSep = 30;
var halfLineSep = lineSep / 2;
var pebbleSep = 15;						// was 30
var centreX = 230;
var baseY = 120;
var pebbleWidth = 15;					// was 30
var pebbleHeight = 15;					// was 30
var pebbleRadius = pebbleWidth / 2;

var subRows = [];	// 0..24			// map of pebbles
var addRows = [];	// 0..24			// map of pebbles
var rowVal = [];

var pebbles = [];	// array of images
pebbles[0] = -1;	// not used

var maxCols = [11, 11, 11, 11, 11, 0, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11];

var normalize = false;
var normDelta = 4;

var precision = 9;

//////////////////////////////////////////////////////////////////////////////////////////

function pos(x, y) {
	var col = (
		x >= centreX
		? Math.floor((x - (centreX + pebbleSep - pebbleRadius)) / pebbleSep) + 1
		: Math.floor((x - (centreX - pebbleSep - pebbleRadius)) / pebbleSep) - 1
	);
	var row = Math.floor((y - baseY + halfLineSep) / lineSep);

	return [col, row];
}

function evalDecimal() {
	var exponent = 0;
	var fraction = 0;
	var i;

	for (i = 0; i < unitRow - 1; i += 1) {
		exponent += rowVal[i] * addRows[i][0];
		exponent -= rowVal[i] * subRows[i][0];
	}

	for (i = unitRow; i <= lastRow; i += 1) {
		fraction += rowVal[i] * addRows[i][0];
		fraction -= rowVal[i] * subRows[i][0];
	}

	//alert("fraction: " + fraction + ", exponent: " + exponent);

	return fraction * Math.pow(10, exponent);
}

function onPebbleMouseDown(event) {
	var gX = event.clientX + window.scrollX - left - foreground.hOffset;
	var gY = event.clientY + window.scrollY - vtop - foreground.vOffset;
	var posn = pos(gX, gY);
	var col = posn[0];
	var row = posn[1];
	var index = this.index;
	var value;

	document.body.removeChild(pebbles[index].div);
	pebbles[index] = undefined;
	if (col > 0) {
		addRows[row][0] -= 1;	// decrement pebble count
		addRows[row][col] = 0;	// remove pebble from the map
	} else {
		subRows[row][0] -= 1;	// decrement pebble count
		subRows[row][-col] = 0;	// remove pebble from the map
	}
	value = evalDecimal();
	output.value = String(value);

	diags.value = "pebble " + index + ": (" + col + ", " + row + ")";
}

function makePebble(i, x, y) {
	var posn = pos(x, y);
	var col = posn[0];
	var row = posn[1];
	var color = (
		x >= centreX
		? "black"
		: "red"
	);

	if (col === 0) {	// avoid centre axis
		return;
	}
	if ((col > 0) && (addRows[row][col] !== 0)) {	// avoid duplicate allocation
		return;
	}
	if ((col < 0) && (subRows[row][-col] !== 0)) {	// avoid duplicate allocation
		return;
	}

	pebbles[i] = newImage(x - pebbleRadius, y - pebbleRadius, pebbleWidth, pebbleHeight, base + color + "_pebble.png", 3, 1.0);
	pebbles[i].index = i;
	pebbles[i].onmousedown = onPebbleMouseDown;

	if (col > 0) {
		addRows[row][0] += 1;	// increment pebble count
		addRows[row][col] = i;	// add pebble to the map
		if (addRows[row][0] >= maxCols[row] - normDelta) {
			normalize = true;
		}
	} else {
		col = -col;
		subRows[row][0] += 1;	// increment pebble count
		subRows[row][col] = i;	// add pebble to the map
		if (subRows[row][0] >= maxCols[row] - normDelta) {
			normalize = true;
		}
	}
}

foreground.onmousedown = function (event) {
	var gX = event.clientX + window.scrollX - left - foreground.hOffset;
	var gY = event.clientY + window.scrollY - vtop - foreground.vOffset;
	var posn = pos(gX, gY);
	var col = posn[0];
	var row = posn[1];
	var value;
	var maxCol = maxCols[row];

	if ((col !== 0) && (col <= maxCol) && (col >= -maxCol) && (row >= 0) && (row <= lastRow)) {
		diags.value = "pebble " + pebbles.length + ": (" + col + ", " + row + ")";
		makePebble(pebbles.length, pebbleSep * col + centreX, lineSep * row + baseY);
		value = evalDecimal();
		output.value = String(value);
		//input.value = toRoman(value);
	}
};

//////////////////////////////////////////////////////////////////////////////////////////

function moveR(p, row, col) {
	var hOffset = col * pebbleSep + centreX - pebbleRadius;
	var vOffset = row * lineSep + baseY - pebbleRadius;

	moveObj(p, hOffset, vOffset);
}

function moveRight() {
	var row;
	var col;
	var fcol;
	var p;

	for (row = 0; row <= lastRow; row += 1) {
		col = 1;
		p = addRows[row][col];

		while (p !== 0) {
			fcol = addRows[row].lastIndexOf(0);
			if ((fcol !== -1) && (fcol > col)) {
				moveR(pebbles[p], row, fcol);
				addRows[row][fcol] = p;
				addRows[row][col] = 0;
			} else {
				break;
			}
			col += 1;
			p = addRows[row][col];
		}
	}
}

function moveL(p, row, col) {
	var hOffset = centreX - col * pebbleSep - pebbleRadius;
	var vOffset = row * lineSep + baseY - pebbleRadius;

	moveObj(p, hOffset, vOffset);
}

function moveLeft() {
	var row;
	var col;
	var fcol;
	var p;

	for (row = 0; row <= lastRow; row += 1) {
		col = 1;
		p = subRows[row][col];

		while (p !== 0) {
			fcol = subRows[row].lastIndexOf(0);
			if ((fcol !== -1) && (fcol > col)) {
				moveL(pebbles[p], row, fcol);
				subRows[row][fcol] = p;
				subRows[row][col] = 0;
			} else {
				break;
			}
			col += 1;
			p = subRows[row][col];
		}
	}
}

splayButton.onclick = function () {
	moveRight();
	moveLeft();
};

//////////////////////////////////////////////////////////////////////////////////////////

var OZ = 1 / 12;
var SU = 1 / 24;
var DS = 1 / 144;
var SC = 1 / 288;
var SI = 1 / 1728;
//var SSI = 1 / 3456;

//983 400 000 => ((C))((M))((L))((X))((X))((X))(M)(M)(M)(C)(D)

var val = {"((M))": 1e9, "((D))": 5e8, "((C))": 1e8, "((L))": 5e7, "((X))": 1e7, "((V))": 5e6, "(M)": 1e6, "(D)": 5e5, "(C)": 1e5, "(L)": 5e4, "(X)": 1e4, "(V)": 5e3, "M": 1000, "D": 500, "C": 100, "L": 50, "X": 10, "V": 5, "I": 1, "S": 0.5, "●": OZ, "£": SU, "$": DS, "℈": SC, "»": SI};
var map = {"((M))": 0, "((D))": 1, "((C))": 2, "((L))": 3, "((X))": 4, "((V))": 5, "(M)": 6, "(D)": 7, "(C)": 8, "(L)": 9, "(X)": 10, "(V)": 11, "M": 12, "D": 13, "C": 14, "L": 15, "X": 16, "V": 17, "I": 18, "S": 19, "●": 20, "£": 21, "$": 22, "℈": 23, "»": 24};

function placePebbles(r, negve, expo) {
	var last = r.length - 1;
	var vali;
	var row;
	var col;
	var mapC = map.C;

	function error(row) {
		alert("Error: No space in row " + row);
	}

	r.forEach(function (ele, i) {
		vali = val[ele];

		if (expo) {
			row = map[ele] - mapC;
		} else {
			row = map[ele] + unitRow;
		}

		if ((i < last) && (vali < val[r[i + 1]])) {
			if (negve) {
				col = addRows[row].indexOf(0, 1);
				if (col === -1) {
					error(row);
					return;
				}
			} else {
				col = subRows[row].indexOf(0, 1);
				if (col === -1) {
					error(row);
					return;
				}
				col = -col;
			}
		} else {
			if (negve) {
				col = subRows[row].indexOf(0, 1);
				if (col === -1) {
					error(row);
					return;
				}
				col = -col;
			} else {
				col = addRows[row].indexOf(0, 1);
				if (col === -1) {
					error(row);
					return;
				}
			}
		}
		makePebble(pebbles.length, pebbleSep * col + centreX, lineSep * row + baseY);
	});
}

function fromRoman2(r, check, negve, expo) {	// converts a [compact] Roman number into a decimal number
	"use strict";						// if check is true,
										// returns null if r is not compact or not in (0..40000000)
										// calls placePebbles to place pebbles on the abacus
	var n = 0;
	var vali;
	var last;

	if (r[0] === "-") {
		r = r.substring(1);
		negve = !negve;
	}

	r = r.toUpperCase();
	if (check && !goodRoman(r)) {
		return null;
	}

	if (!compactBox.checked) {
		r = uncompact(r);
	}
    r = splitRoman(r);

	if (typeof r === "number") {
		alert("Error at position " + r);
		return null;	// was NaN;
	}

	last = r.length - 1;

	r.forEach(function (ele, i) {
		vali = val[ele];

		if ((i < last) && (vali < val[r[i + 1]])) {
			n -= vali;
		} else {
			n += vali;
		}
	});

	placePebbles(r, negve, expo);

	return n;
}

//////////////////////////////////////////////////////////////////////////////////////////

function clearRows() {	// clear pebble counts and create the pebble map
	var i;
	var j;
	var maxCol;

	for (i = 0; i <= lastRow; i += 1) {
		addRows[i] = [];
		subRows[i] = [];

		addRows[i][0] = 0;					// clear pebble count
		subRows[i][0] = 0;					// clear pebble count

		maxCol = maxCols[i];

		for (j = 1; j <= maxCol; j += 1) {	// clear the pebble map
			addRows[i][j] = 0;				// clear pebble index
			subRows[i][j] = 0;				// clear pebble index
		}
	}
}

function clearPebbles() {
	var i;

	for (i = 1; i < pebbles.length; i += 1) {
		if ((pebbles[i] !== undefined) && (pebbles[i].div !== undefined)) {
			document.body.removeChild(pebbles[i].div);
		}
	}
	pebbles = [];
	pebbles[0] = -1;	// not used
	clearRows();
}

function warn() {
	if (normalize) {
		alert("Please normalize!");
		diags.value = "Please normalize!";
	}
	normalize = false;
}

function decompose(x) {	 // decompose Number x into { sign, exponent, fraction }
	var sigN = "";
	var exponenT = 0; // sign 1 or -1, base 10 exponent

	if (x < 0) {
		x = -x;
		sigN = "-";
	}

	if (x < 1e-300) {
		return {
			sign: "",
			exponent: 0,
			fraction: 0
		};
	}

	while (x >= 1) {
		x = 0.1 * x;
		exponenT += 1;
	}
	while (x < 0.1) {
		x = 10 * x;
		exponenT -= 1;
	}
	return {
		sign: sigN,
		exponent: exponenT,
		fraction: x
	};
}

function process(obj, shiftSet) {
	var value;
	var n;
	var sign;
	var exponent;
	var fraction;
	var fValue;
	var eValue;

	if (obj.value === "") {
		clearPebbles();
		return;
	}
	value = Number(obj.value);
	if (Number.isNaN(value)) {
		diags.value = "Not a valid number.";
		return;
	}

	n = decompose(value);
	sign = n.sign;
	exponent = n.exponent;
	fraction = n.fraction.toFixed(9);
	if (fraction === 0) {
		clearPebbles();
		return;
	}
	fValue = toRoman(Number(sign + fraction.substring(2)));
	clearPebbles();
	value = fromRoman2(fValue, true, shiftSet, false);
	eValue = toRoman(Number(exponent));
	value = fromRoman2(eValue, true, shiftSet, true);
	value = evalDecimal();
	diags.value = value.toPrecision(precision);
	warn();
}

input.onkeydown = function (event) {
	var item = event.which || event.keyCode;

	if (item === 13) {
		process(input, false);
	}
};

output.onkeydown = function (event) {
	var item = event.which || event.keyCode;
	var altSet = event.altKey || altKey.checked;
	var shiftSet = event.shiftKey || shiftKey.checked;
	var tmp;

	if (item === 13) {
		if (shiftSet) {
			// subtract output from input
			tmp = parseFloat(input.value) - parseFloat(output.value);
			input.value = String(tmp);
			process(input, false);
			input.value = tmp.toPrecision(precision);
		} else if (altSet) {
			// add output to input
			tmp = parseFloat(input.value) + parseFloat(output.value);
			input.value = String(tmp);
			process(input, false);
			input.value = tmp.toPrecision(precision);
		} else {
			process(output, false);
		}
	}
};

//////////////////////////////////////////////////////////////////////////////////////////

(function () {	// initialize rowVal
	var i;

	for (i = 0; i < maxPower; i += 1) {
		rowVal[2 * i] = Math.pow(10, maxPower - i);
		rowVal[2 * i + 1] = rowVal[2 * i] / 2;
	}
	rowVal[lastRowExp] = 1;

	for (i = maxPower + 1; i < minPower; i += 1) {
		rowVal[2 * i] = Math.pow(10, maxPower + 1 - i);
		rowVal[2 * i + 1] = rowVal[2 * i] / 2;
	}

	rowVal[lastRow] = Math.pow(10, maxPower + 1 - minPower);
}());


function sample() {
	var value = "9.834e15";

	input.value = "0";
	output.value = value;
	process(output, false);
}

//////////////////////////////////////////////////////////////////////////////////////////

clearRows();
sample();	// 0.9834 x 10 ^16
