/*
	Series_To_Parallel_Conversion_Calculator
 	Copyright © 2020 Harry Whitfield

 	This program is free software; you can redistribute it and/or modify it
 	under the terms of the GNU General Public License as published by the
 	Free Software Foundation; either version 2 of the License, or (at your
 	option) any later version.

 	This program is distributed in the hope that it will be useful, but
 	WITHOUT ANY WARRANTY; without even the implied warranty of
 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 	General Public License for more details.

 	You should have received a copy of the GNU General Public License along
 	with this program; if not, write to the Free Software Foundation, Inc.,
 	51 Franklin St, Fifth Floor, Boston, MA	 02110-1301	 USA

 	Series_To_Parallel_Conversion_Calculator - browser version 1.2
 	16 June, 2020
 	Copyright © 2020 Harry Whitfield
 	mailto:g6auc@arrl.net

	This version has been modified to facilitate future enhancements.
*/

/*jslint browser, devel, for, this */

/*property
    G, M, PI, T, altKey, background, backgroundColor, focus, indexOf, isNaN, k,
    keyCode, length, m, match, n, onkeydown, p, pow, preventDefault, replace,
    select, style, substring, title, toFixed, toString, u, value, which
*/

import {paramOne, paramTwo, paramThree, paramFour, paramFive} from "./model.js";
import {headingOne, headingTwo, headingThree, headingFour, headingFive} from "./model.js";
import {title, execButton} from "./model.js";

//var gWidgetName;	// global

// map custom names onto model names
var sR = paramOne;
var sX = paramTwo;
var pR = paramThree;
var pX = paramFour;
var hZ = paramFive;

var sRHeading = headingOne;
var sXHeading = headingTwo;
var pRHeading = headingThree;
var pXHeading = headingFour;
var hZHeading = headingFive;

//var title;
title.value = "Series to Parallel Conversion";

var rSuffixes = "\nYou may append one of the SI suffixes";
rSuffixes += "\nT, G, M, k, m, µ/u, n or p";
rSuffixes += "\nfollowed optionally by an Ω character.";
rSuffixes += "\n\nPress RETURN to calculate.";

var xSuffixes = "\nYou may append one of the SI suffixes";
xSuffixes += "\nT, G, M, k, m, µ/u, n or p";
xSuffixes += "\nfollowed by one of the characters H, F";
xSuffixes += "\nor an optional Ω character.";
xSuffixes += "\n\nPress RETURN to calculate.";

var fSuffixes = "\nYou may append one of the SI suffixes";
fSuffixes += "\nT, G, M, k, m, µ/u, n or p";
fSuffixes += "\nfollowed by the characters Hz.";
fSuffixes += "\n\nPress RETURN to validate.";

sR.title = "Enter the series resistance." + rSuffixes;
sX.title = "Enter the series reactance." + xSuffixes;
pR.title = "Enter the parallel resistance." + rSuffixes;
pX.title = "Enter the parallel reactance." + xSuffixes;
hZ.title = "Enter the frequency." + fSuffixes;

sRHeading.value = "Series Resistance";
sXHeading.value = "Series Reactance";
pRHeading.value = "Parallel Resistance";
pXHeading.value = "Parallel Reactance";
hZHeading.value = "Frequency";

var s2p = true;
var prec = 3;
var freq = null;
var badFreq = false;

function cToX(f, c) {
	return 1 / (2 * Math.PI * f * c);
}

function lToX(f, l) {
	return 2 * Math.PI * f * l;
}

function xToC(f, x) {
	return 1 / (2 * Math.PI * f * x);
}

function xToL(f, x) {
	return x / (2 * Math.PI * f);
}

function reduce(s) {
	var idx = s.indexOf(" == ");

	if (idx !== -1) {
		s = s.substring(0, idx);
	}
	return s.replace(/\s/g, "");
}

function decodeSI(s) {	// now allows for a unit suffix F|Hz|H|Ω
	const table = {"T": 1e12, "G": 1e9, "M": 1e6, "k": 1e3, "Ω": 1, "m": 1e-3, "µ": 1e-6, "u": 1e-6, "n": 1e-9, "p": 1e-12};
	const ok = /^-?\d+(\.\d+)?([eE]-?\d{1,2})?[TGMkmµunp]?(F|Hz|H|Ω)?$/;
	var found;
	var unit;	// unit suffix
	var uLen = 0;
	var val;
	var suffix;	// multiplier suffix
	var power;

	s = reduce(s);

	if (s === "Infinity") {
		return [1e99];
	}
	if (s === "-Infinity") {
		return [-1e99];
	}

	found = s.match(ok);
	if (found === null) {
		return [NaN];
	}

	unit = found[3];
	if (unit !== undefined) {
		uLen = unit.length;
	}

	val = parseFloat(s);
	suffix = s[s.length - 1 - uLen];
	power = table[suffix];

	if (power !== undefined) {
		val *= power;
	}
	return [val, unit];
}

function encodeSI(n, unit) {
	const table = ["T", "G", "M", "k", "", "m", "µ", "n", "p"];
	var i;
	var j;
	var p;
	var tmp;
	var sign = "";
	var res;

	unit = unit || "Ω";

	if (n === Infinity) {
		return Infinity;
	}

	if (n < 0) {
		sign = "-";
		n = -n;
	}

	for (i = 4; i >= -4; i -= 1) {
		j = 3 * i;
		p = Math.pow(10, j);
		if (n >= p) {
			tmp = n / p;
			res = tmp.toFixed(prec) + table[4 - i] + unit;
			return (
				sign === "-"
				? sign + res
				: res
			);
		}
	}
	return (
		sign === "-"
		? -n.toString() + unit
		: n.toString() + unit
	);
}

function process() {
    var rS;
    var rU;
    var xS;
    var xU;
    var xL;
    var xC;
    var rP;
    var xP;
    var ms;
    var res;

	if (s2p) {
		res = decodeSI(sR.value);
		rS = res[0];
		rU = res[1];
		res = decodeSI(sX.value);
		xS = res[0];
		xU = res[1];

		if (rU && (rU !== "Ω")) {
        	pR.value = "Invalid Series Resistance";
        	return;
        }
   		if (Number.isNaN(rS)) {
        	pR.value = "Invalid Series Resistance";
        	return;
    	}
    	if (rS < 0) {
    		pR.value = "Invalid Series Resistance";
    		return;
    	}

		if (xU && (xU !== "Ω") && (xU !== "H") && (xU !== "F")) {
        	pX.value = "Invalid Series Reactance";
        	return;
        }
    	if (Number.isNaN(xS)) {
        	pX.value = "Invalid Series Reactance";
        	return;
    	}

    	if (xU === "H") {
    		if (xS < 0) {
    			pX.value = "Invalid Series Reactance";
    			return;
    		}
    		if (freq === null) {
     			pX.value = "Please set a frequency!";
    			return;
    		}
    	 	xL = xS;
    	 	xS = lToX(freq, xS);
    	 	sX.value = encodeSI(xS, "Ω") + " == " + encodeSI(xL, "H");
    	} else if (xU === "F") {
    		if (xS < 0) {
    			pX.value = "Invalid Series Reactance";
    			return;
    		}
    		if (freq === null) {
     			pX.value = "Please set a frequency!";
    			return;
    		}
    	 	xC = xS;
    	 	xS = -cToX(freq, xS);
    	 	sX.value = encodeSI(xS, "Ω") + " == " + encodeSI(xC, "F");
    	} else if (freq !== null) {
    		if (xS < 0) {
    			xC = xToC(freq, -xS);
    			sX.value = encodeSI(xS, "Ω") + " == " + encodeSI(xC, "F");
    		} else {
				xL = xToL(freq, xS);
				sX.value = encodeSI(xS, "Ω") + " == " + encodeSI(xL, "H");
    		}
    	}

	  	ms = rS * rS + xS * xS;
     	pR.value = encodeSI(ms / rS);
    	pX.value = encodeSI(ms / xS);

    	if (freq !== null) {
    		if (xS < 0) {
    			pX.value += " == " + encodeSI(xToC(freq, -ms / xS), "F");
    		} else {
    			pX.value += " == " + encodeSI(xToL(freq, ms / xS), "H");
     		}
    	}
    } else {
		res = decodeSI(pR.value);
		rP = res[0];
		rU = res[1];
		res = decodeSI(pX.value);
		xP = res[0];
		xU = res[1];

		if (rU && (rU !== "Ω")) {
        	sR.value = "Invalid Parallel Resistance";
        	return;
        }
    	if (Number.isNaN(rP)) {
        	sR.value = "Invalid Parallel Resistance";
        	return;
    	}
    	if (rP < 0) {
    		sR.value = "Invalid Parallel Resistance";
    		return;
    	}

		if (xU && (xU !== "Ω") && (xU !== "H") && (xU !== "F")) {
        	sX.value = "Invalid Parallel Reactance";
        	return;
        }
    	if (Number.isNaN(xP)) {
        	sX.value = "Invalid Parallel Reactance";
        	return;
    	}

    	if (xU === "H") {
    		if (xP < 0) {
    			sX.value = "Invalid Series Reactance";
    			return;
    		}
    		if (freq === null) {
     			sX.value = "Please set a frequency!";
    			return;
    		}
    	 	xL = xP;
    	 	xP = lToX(freq, xP);
    	 	pX.value = encodeSI(xP, "Ω") + " == " + encodeSI(xL, "H");
    	} else if (xU === "F") {
    		if (xP < 0) {
    			sX.value = "Invalid Series Reactance";
    			return;
    		}
    		if (freq === null) {
     			sX.value = "Please set a frequency!";
    			return;
    		}
     	 	xC = xP;
    	 	xP = -cToX(freq, xP);
    	 	pX.value = encodeSI(xP, "Ω") + " == " + encodeSI(xC, "F");
    	} else if (freq !== null) {
    		if (xP < 0) {
    			xC = xToC(freq, -xP);
    			pX.value = encodeSI(xP, "Ω") + " == " + encodeSI(xC, "F");
    		} else {
				xL = xToL(freq, xP);
				pX.value = encodeSI(xP, "Ω") + " == " + encodeSI(xL, "H");
    		}
    	}

		ms = rP * rP + xP * xP;
		sR.value = encodeSI(rP * xP * xP / ms);
		sX.value = encodeSI(xP * rP * rP / ms);

    	if (freq !== null) {
    		if (xP < 0) {
    			sX.value += " == " + encodeSI(xToC(freq, -xP * rP * rP / ms), "F");
    		} else {
    			sX.value += " == " + encodeSI(xToL(freq, xP * rP * rP / ms), "H");
     		}
    	}
    }
}

function setBackgrounds() {
	const lite = "rgba(255 ,255, 255, 1.0)";
	const dark = "rgba(192 ,192, 192, 1.0)";

	if (s2p) {
		sR.style.background = lite;
		sX.style.background = lite;
		pR.style.background = dark;
		pX.style.background = dark;
	} else {
		sR.style.background = dark;
		sX.style.background = dark;
		pR.style.background = lite;
		pX.style.background = lite;
	}
}

function reset(o) {
	var s = o.value;
	var idx = s.indexOf(" == ");

	if (idx !== -1) {
		o.value = s.substring(0, idx);
	}
}

function setFrequency() {
	const paleGreen = "rgba(0, 255, 0, 0.5)";
	var zH;
	var zU;
	var res;

	if (hZ.value === "") {
        freq = null;
    	if (badFreq) {
	    	pX.value = "";
	    	badFreq = false;
		}
        return;
	}

	res = decodeSI(hZ.value);
	zH = res[0];
	zU = res[1];

	if (zU && (zU !== "Hz")) {
        pX.value = "Invalid Frequency";
        freq = null;
        badFreq = true;
        return;
    }
   	if (Number.isNaN(zH)) {
        pX.value = "Invalid Frequency";
        freq = null;
        badFreq = true;
        return;
    }
    if (zH <= 0) {
    	pX.value = "Invalid Frequency";
        freq = null;
        badFreq = true;
        return;
    }

	freq = zH;

	hZ.style.backgroundColor = paleGreen;

    if (badFreq) {
	    pX.value = "";
	    badFreq = false;
	}
}

function test_case() {
	const paleGreen = "rgba(0, 255, 0, 0.5)";

	s2p = true;
	setBackgrounds();
	sR.value = "10Ω";
	sX.value = "-20Ω";

	freq = 1e7;
	hZ.value = "10MHz";
	hZ.style.backgroundColor = paleGreen;
	process();
}

var Tab = 9;
var Return = 13;
var keyM = 77;	// for µ
var keyZ = 90;	// for Ω

sR.onkeydown = function (event) {
	var x = event.which || event.keyCode;

	if (x === Tab) {
		event.preventDefault();
		sX.focus();
		sX.select();
		return;
	}

	s2p = true;
	setBackgrounds();

	if (x === Return) {
		process();
    } else {
        pR.value = "";
        pX.value = "";
    }

    if (event.altKey && (x === keyZ)) {
    	event.preventDefault();
    	this.value += "Ω";
    }

    if (event.altKey && (x === keyM)) {
    	event.preventDefault();
    	this.value += "µ";
    }
};

sX.onkeydown = function (event) {
	var x = event.which || event.keyCode;

	if (x === Tab) {
		event.preventDefault();
		pR.focus();
		pR.select();
		return;
	}

	s2p = true;
	setBackgrounds();

//	reset(this);

	if (x === Return) {
		process();
    } else {
        pR.value = "";
        pX.value = "";
    }

    if (event.altKey && (x === keyZ)) {
    	event.preventDefault();
    	this.value += "Ω";
    }

    if (event.altKey && (x === keyM)) {
    	event.preventDefault();
    	this.value += "µ";
    }

    if (event.altKey && (x === keyZ)) {
    	event.preventDefault();
    	this.value += "Ω";
    }

    if (event.altKey && (x === keyM)) {
    	event.preventDefault();
    	this.value += "µ";
    }
};

pR.onkeydown = function (event) {
	var x = event.which || event.keyCode;

	if (x === Tab) {
		event.preventDefault();
		pX.focus();
		pX.select();
		return;
	}

	s2p = false;
	setBackgrounds();

	if (x === Return) {
		process();
    } else {
        sR.value = "";
        sX.value = "";
    }

    if (event.altKey && (x === keyZ)) {
    	event.preventDefault();
    	this.value += "Ω";
    }

    if (event.altKey && (x === keyM)) {
    	event.preventDefault();
    	this.value += "µ";
    }
};

pX.onkeydown = function (event) {
	var x = event.which || event.keyCode;

	if (x === Tab) {
		event.preventDefault();
		sR.focus();
		sR.select();
		return;
	}

	s2p = false;
	setBackgrounds();

//	reset(this);

	if (x === Return) {
		process();
    } else {
        sR.value = "";
        sX.value = "";
    }

    if (event.altKey && (x === keyZ)) {
    	event.preventDefault();
    	this.value += "Ω";
    }

    if (event.altKey && (x === keyM)) {
    	event.preventDefault();
    	this.value += "µ";
    }
};

hZ.onkeydown = function (event) {
	var x = event.which || event.keyCode;

	if (x === Tab) {
		event.preventDefault();
		sR.focus();
		sR.select();
		return;
	}

	reset(sX);
	reset(pX);

	hZ.style.backgroundColor = "transparent";
	freq = null;

	if (x === Return) {
		setFrequency();
    }

    if (event.altKey && (x === keyZ)) {
    	event.preventDefault();
    	this.value += "Ω";
    }

    if (event.altKey && (x === keyM)) {
    	event.preventDefault();
    	this.value += "µ";
    }
};

execButton.onmousedown = function () {
	execButton.style.opacity = "0.5";
};
execButton.onmouseup = function () {
	execButton.style.opacity = "1.0";
	process();
};

//gWidgetName = "Series_To_Parallel_Conversion Web Widget\n\n";
test_case();
