/*
	Tx Line - A Transmission Line Calculator
	Copyright © 2007-2015 Harry Whitfield

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2 of the License, or (at your
	option) any later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.

	You should have received a copy of the GNU General Public License along
	with this program; if not, write to the Free Software Foundation, Inc.,
	51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

	Tx Line - browser version 1.0
	26 April, 2015
	Copyright © 2007-2015 Harry Whitfield
	mailto:g6auc@arrl.net
*/

/*global paramOne, paramTwo, paramThree, paramFour,
	headingOne, headingTwo, headingThree, headingFour, clearData, Menu
*/

/*properties
    PI, background, exp, keyCode, log, onchange, onkeypress, selectedIndex, sqrt,
    style, title, toFixed, value, which
*/

// map custom names onto model names
var outerRadius     = paramOne;
var innerRadius     = paramTwo;
var relPermittivity = paramThree;
var impedance       = paramFour;

var outerRadiusHeading     = headingOne;
var innerRadiusHeading     = headingTwo;
var relPermittivityHeading = headingThree;
var impedanceHeading       = headingFour;

var outputModeMenu = Menu[0];
var iModeMenu      = Menu[1];

var mode     = 0;	// Outer Radius
var iMode    = 0;	// Coaxial Cable
/*
var modeText    = ['Outer Radius',    'Inner Radius', 'Relative Permittivity', 'Characteristic Impedance'];
var altModeText = ['Line Separation', 'Line Radius',  'Relative Permittivity', 'Characteristic Impedance'];
var iModeText   = ['Coaxial Cable', 'Parallel Line'];
*/
var modeValue = 0;
var oldModeValue = 0;
var iModeValue = 0;

var multiplier;

function parseNonNegFloat(s) {
    var v = parseFloat(s);
    if (isNaN(v)) {
        return v;
    }
    if (v >= 0) {
        return v;
    }
    return NaN;
}

function parseDielecFloat(s) {
    var v = parseFloat(s);
    if (isNaN(v)) {
        return v;
    }
    if (v >= 1) {
        return v;
    }
    return NaN;
}

function process() {
    var mu0 = 4 * Math.PI * 1e-7, 	// Permeability of Vacuum
    	c = 299792458, 				// Speed of light in Vacuum
//    	e0 = 1 / (mu0 * c * c), 	// Permittivity of Vacuum
    	Z0 = c * mu0, 				// Characteristic Impedance of Vacuum
		outer,
        inner,
        e,
        Z,
        result;

    function charImp(o, i, e) { // Characteristic Impedance of Line
        return Z0 * Math.log(o / i) / (multiplier * Math.PI * Math.sqrt(e));
    }

    function relPerm(o, i, Z) {
        var sqrtE = Z0 * Math.log(o / i) / (multiplier * Math.PI * Z);
        return sqrtE * sqrtE;
    }

    function innerRad(o, e, Z) {
        return o / Math.exp(Z * multiplier * Math.PI * Math.sqrt(e) / Z0);
    }

    function outerRad(i, e, Z) {
        return i * Math.exp(Z * multiplier * Math.PI * Math.sqrt(e) / Z0);
    }


    switch (mode) { // Z=8, e=4, inner=2, ouuter=1
    case 3:
        outer = parseNonNegFloat(outerRadius.value);
        inner = parseNonNegFloat(innerRadius.value);
        e = parseDielecFloat(relPermittivity.value);
        result = charImp(outer, inner, e);
        //print('result: ' + result);
        if (isNaN(result)) {
            impedance.value = 'Invalid Input';
        } else {
        	impedance.value = result.toFixed(4);
        }
        break;

    case 2:
        outer = parseNonNegFloat(outerRadius.value);
        inner = parseNonNegFloat(innerRadius.value);
        Z = parseNonNegFloat(impedance.value);
        result = relPerm(outer, inner, Z);
        //print('result: ' + result);
        if (isNaN(result)) {
            relPermittivity.value = 'Invalid Input';
        } else {
        	relPermittivity.value = result.toFixed(4);
        }
        break;

    case 1:
        outer = parseNonNegFloat(outerRadius.value);
        e = parseDielecFloat(relPermittivity.value);
        Z = parseNonNegFloat(impedance.value);
        result = innerRad(outer, e, Z);
        //print('result: ' + result);
        if (isNaN(result)) {
            innerRadius.value = 'Invalid Input';
        } else {
        	innerRadius.value = result.toFixed(4);
        }
        break;

    case 0:
        inner = parseNonNegFloat(innerRadius.value);
        e = parseDielecFloat(relPermittivity.value);
        Z = parseNonNegFloat(impedance.value);
        result = outerRad(inner, e, Z);
        //print('result: ' + result);
        if (isNaN(result)) {
            outerRadius.value = 'Invalid Input';
        } else {
        	outerRadius.value = result.toFixed(4);
        }
        break;

    default:
//      beep();
        break;
    }
}

function updatePrefs() {
	mode  = modeValue;
	iMode = iModeValue;

    if (iMode === 0) { // coax
        multiplier = 2.0;
        outerRadiusHeading.value = "Outer Radius";
        innerRadiusHeading.value = "Inner Radius";
        outerRadius.title = "Outer radius of the coaxial cable (i.e. the radius of the inner surface of the shield).";
        innerRadius.title = "Inner radius of the coaxial cable (i.e. the radius of the outer surface of the central conductor).";
//        chooseMenu.reset(modeText, modeText[modeIndex[mode]]);
//        preferences.modePref.option = modeText;
//        buildVitality("Resources/coax.png", 224, 200, dockText[lineType]);
    } else {
        multiplier = 1.0;
        outerRadiusHeading.value = "Line Separation";
        innerRadiusHeading.value = "Line Radius";
        outerRadius.title = "Line Separation (i.e. the distance between the centers of the conductors).";
        innerRadius.title = "Line Radius (i.e. the radius of the outer surface of each conductor).";
//        chooseMenu.reset(altModeText, altModeText[modeIndex[mode]]);
//        preferences.modePref.option = altModeText;
//        buildVitality("Resources/line.png", 224, 200, dockText[lineType]);
    }
}

outputModeMenu.onchange  = function () {
	modeValue = this.selectedIndex;
	switch (oldModeValue) {
	case 0:
		outerRadius.style.background = "rgba(255, 255, 255, 1.0)";
		break;
	case 1:
		innerRadius.style.background = "rgba(255, 255, 255, 1.0)";
		break;
	case 2:
		relPermittivity.style.background = "rgba(255, 255, 255, 1.0)";
		break;
	case 3:
		impedance.style.background = "rgba(255, 255, 255, 1.0)";
		break;
	}

	switch (modeValue) {
	case 0:
		outerRadius.style.background = "rgba(192, 192, 192, 1.0)";
		break;
	case 1:
		innerRadius.style.background = "rgba(192, 192, 192, 1.0)";
		break;
	case 2:
		relPermittivity.style.background = "rgba(192, 192, 192, 1.0)";
		break;
	case 3:
		impedance.style.background = "rgba(192, 192, 192, 1.0)";
		break;
	}
	
	oldModeValue = modeValue;
	updatePrefs();
};

iModeMenu.onchange  = function () {
	iModeValue = this.selectedIndex;
	updatePrefs();
};

outerRadius.onkeypress = function (event) {
	var x = event.which || event.keyCode;
	
	if (x === 13) { process(); }
};

innerRadius.onkeypress = function (event) {
	var x = event.which || event.keyCode;
	
	if (x === 13) { process(); }
};

relPermittivity.onkeypress = function (event) {
	var x = event.which || event.keyCode;
	
	if (x === 13) { process(); }
};

impedance.onkeypress = function (event) {
	var x = event.which || event.keyCode;
	
	if (x === 13) { process(); }
};

//updatePrefs();
