/*
    Wiring Calculator
    Copyright © 2010,2015 Harry Whitfield

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    Wiring Calculator - browser version 1.0
    7 July, 2015
    Copyright © 2010,2015 Harry Whitfield
    mailto:g6auc@arrl.net
*/

/*jslint browser, for, multivar */

/*global paramOne, paramTwo, paramThree, paramFour,
    headingOne, headingTwo, headingThree, headingFour, clearData, Menu, alert
*/

/*property
    background, keyCode, length, onchange, onkeypress, selectedIndex, style,
    title, toFixed, value, which
*/

// map custom names onto model names
var circuitLength = paramOne;
var current = paramTwo;
var wireArea = paramThree;
var voltageDrop = paramFour;

var circuitLengthHeading = headingOne;
var currentHeading = headingTwo;
var wireAreaHeading = headingThree;
var voltageDropHeading = headingFour;

var outputModeMenu = Menu[0];
var unitsMenu = Menu[1];
var temperatureMenu = Menu[2];
var deratingConstantMenu = Menu[3];

var mode = 0;   // Circuit Length
var units = 0;  // SI
/*
var modeText = ['Circuit Length', 'Current', 'Cross Section', 'Voltage Drop'];
var lineText = ['SI', 'USA'];
*/
var modeValue = 0;
var oldModeValue = 0;
var unitsValue = 0;
var temperatureValue = 20;
var deratingConstantValue = 100;

var lengthMultiplier;
var deratingMultiplier;

var rho = 2.0e-8;   // resistivity of copper at 70 degrees Celsius.

var mmsqOfAwg = [53.5, 42.4, 33.6, 26.7, 21.2, 16.8, 13.3, 10.5, 8.37, 6.63,
        5.26, 4.17, 3.31, 2.62, 2.08, 1.65, 1.31, 1.04, 0.823, 0.653,
        0.518];

var ampsOfAwg = [170, 150, 130, 110, 95, 85, 75, 65, 55, 47,
        40, 35, 30, 27, 25, 21, 18, 16, 14, 12,
        11];

function parseNonNegFloat(s) {
    "use strict";
    var v = parseFloat(s);

    if (isNaN(v)) {
        return v;
    }
    if (v >= 0) {
        return v;
    }
    return NaN;
}

function parseArea(s) {
    "use strict";
    var awg, mmsq;

    if (units === 0) {
        return parseNonNegFloat(s);
    }
    awg = parseInt(s, 10);
    if (isNaN(awg)) {
        return NaN;
    }
    mmsq = mmsqOfAwg[awg];
    if (mmsq === undefined) {
        return NaN;
    }
    return mmsq;
}

function convertArea(v) {
    "use strict";
    var i;

    if (units === 0) {
        return v.toFixed(2);
    }
    for (i = mmsqOfAwg.length - 1; i >= 0; i -= 1) {
        if (mmsqOfAwg[i] >= v) {
            return i;
        }
    }
    return "Invalid Input";
}

function ampacity(v) {
    "use strict";
    var i;

    if (v > 53.5) {
        return -1;
    }
    for (i = 0; i < mmsqOfAwg.length; i += 1) {
        if (v >= mmsqOfAwg[i]) {
            return deratingMultiplier * ampsOfAwg[i];
        }
    }
    return -1;
}

function process() {
    "use strict";
    var length, amps, area, drop, tmp;   // wpm;

    switch (mode) { // Voltage Drop=8, Cross Section=4, Curent=2, Circuit Length=1
    case 3:
        length = lengthMultiplier * parseNonNegFloat(circuitLength.value);
        amps = parseNonNegFloat(current.value);
        area = 1e-6 * parseArea(wireArea.value);
        drop = rho * length * amps / area;
        //print('drop: ' + drop);
        if (isNaN(drop)) {
            voltageDrop.value = "Invalid Input";
        } else {
            voltageDrop.value = drop.toFixed(3);
            //wpm = drop * amps / length;
            //print('wpm: ' + wpm.toFixed(3));
            if (amps > ampacity(1e6 * area)) {
                //beep();
                alert("Please check the current-carrying capacity of your wire!");
            }
        }
        break;

    case 2:
        length = lengthMultiplier * parseNonNegFloat(circuitLength.value);
        amps = parseNonNegFloat(current.value);
        drop = parseNonNegFloat(voltageDrop.value);
        area = rho * length * amps / drop;
        //print('area: ' + 1e6 * area);
        if (isNaN(area)) {
            wireArea.value = "Invalid Input";
        } else {
            wireArea.value = convertArea(1e6 * area);
            //wpm = drop * amps / length;
            //print('wpm: ' + wpm.toFixed(3));
            if (amps > ampacity(1e6 * area)) {
                //beep();
                alert("Please check the current-carrying capacity of your wire!");
            }
        }
        break;

    case 1:
        length = lengthMultiplier * parseNonNegFloat(circuitLength.value);
        area = 1e-6 * parseArea(wireArea.value);
        drop = parseNonNegFloat(voltageDrop.value);
        amps = area * drop / (rho * length);
        //print('amps: ' + amps);
        if (isNaN(amps)) {
            current.value = "Invalid Input";
        } else {
            current.value = amps.toFixed(2);
            //wpm = drop * amps / length;
            //print('wpm: ' + wpm.toFixed(3));
            if (amps > ampacity(1e6 * area)) {
                //beep();
                alert("Please check the current-carrying capacity of your wire!");
            }
        }
        break;

    case 0:
        amps = parseNonNegFloat(current.value);
        area = 1e-6 * parseArea(wireArea.value);
        drop = parseNonNegFloat(voltageDrop.value);
        length = area * drop / (rho * amps);
        //print('length: ' + length);
        if (isNaN(length)) {
            circuitLength.value = "Invalid Input";
        } else {
            tmp = length / lengthMultiplier;
            circuitLength.value = tmp.toFixed(2);
            //wpm = drop * amps / length;
            //print('wpm: ' + wpm.toFixed(3));
            if (amps > ampacity(1e6 * area)) {
                //beep();
                alert("Please check the current-carrying capacity of your wire!");
            }
        }
        break;

    default:
        //beep();
        break;
    }
}

function updatePrefs() {
    "use strict";
    var t = temperatureValue;
    rho = 1.68 * (1 + 0.0039 * (t - 20));
    //print("rho: " + rho + " x 10^-8");
    rho *= 1e-8;

    mode = modeValue;
    units = unitsValue;

    if (units === 0) {
        lengthMultiplier = 1.0;
        circuitLengthHeading.value = "Circuit Length (m)";
        circuitLength.title = "Circuit Length (in metres).";
        wireAreaHeading.value = "Cross Section (mm^2)";
        wireArea.title = "Cross Section (in square millimetres).";
        //buildVitality("Resources/si.png", 536, 512, dockText[units]);
    } else {
        lengthMultiplier = 0.3048;
        circuitLengthHeading.value = "Circuit Length (ft)";
        circuitLength.title = "Circuit Length (in feet).";
        wireAreaHeading.value = "Wire Size (AWG)";
        wireArea.title = "Wire Size (0..20 AWG).";
        //buildVitality("Resources/us.png", 323, 296, dockText[units]);
    }

    deratingMultiplier = deratingConstantValue / 100;

    //clearData();
}

outputModeMenu.onchange = function () {
    "use strict";
    modeValue = outputModeMenu.selectedIndex;
    switch (oldModeValue) {
    case 0:
        circuitLength.style.background = "rgba(255, 255, 255, 1.0)";
        break;
    case 1:
        current.style.background = "rgba(255, 255, 255, 1.0)";
        break;
    case 2:
        wireArea.style.background = "rgba(255, 255, 255, 1.0)";
        break;
    case 3:
        voltageDrop.style.background = "rgba(255, 255, 255, 1.0)";
        break;
    }

    switch (modeValue) {
    case 0:
        circuitLength.style.background = "rgba(192, 192, 192, 1.0)";
        break;
    case 1:
        current.style.background = "rgba(192, 192, 192, 1.0)";
        break;
    case 2:
        wireArea.style.background = "rgba(192, 192, 192, 1.0)";
        break;
    case 3:
        voltageDrop.style.background = "rgba(192, 192, 192, 1.0)";
        break;
    }

    oldModeValue = modeValue;
    updatePrefs();
};

unitsMenu.onchange = function () {
    "use strict";
    unitsValue = unitsMenu.selectedIndex;
    updatePrefs();
};

temperatureMenu.onchange = function () {
    "use strict";
    temperatureValue = parseInt(temperatureMenu.value, 10);
    updatePrefs();
};

deratingConstantMenu.onchange = function () {
    "use strict";
    deratingConstantValue = parseInt(deratingConstantMenu.value, 10);
    updatePrefs();
};

circuitLength.onkeypress = function (event) {
    "use strict";
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

current.onkeypress = function (event) {
    "use strict";
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

wireArea.onkeypress = function (event) {
    "use strict";
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

voltageDrop.onkeypress = function (event) {
    "use strict";
    var x = event.which || event.keyCode;

    if (x === 13) {
        process();
    }
};

updatePrefs();
